<?php
/**
 * ADMINISTRAÇÃO DE ARTIGOS DE AJUDA - WHITE LABEL v2.0
 * Gerenciar tutoriais e artigos de ajuda para usuários
 */
require_once 'config.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: dashboard.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$sucesso = '';
$erro = '';

// Buscar dados do usuário
$stmt = $pdo->prepare("SELECT * FROM usuarios WHERE id = ?");
$stmt->execute([$user_id]);
$usuario = $stmt->fetch();

// CRUD de Artigos
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $acao = $_POST['acao'] ?? '';

    try {
        if ($acao === 'criar') {
            $stmt = $pdo->prepare("INSERT INTO artigos_ajuda (titulo, conteudo, ordem, visivel_para, publicado) VALUES (?, ?, ?, ?, ?)");
            $stmt->execute([
                $_POST['titulo'],
                $_POST['conteudo'],
                $_POST['ordem'] ?? 0,
                $_POST['visivel_para'] ?? 'todos',
                isset($_POST['publicado']) ? 1 : 0
            ]);
            $sucesso = 'Artigo criado com sucesso!';
        } elseif ($acao === 'editar') {
            $stmt = $pdo->prepare("UPDATE artigos_ajuda SET titulo = ?, conteudo = ?, ordem = ?, visivel_para = ?, publicado = ? WHERE id = ?");
            $stmt->execute([
                $_POST['titulo'],
                $_POST['conteudo'],
                $_POST['ordem'] ?? 0,
                $_POST['visivel_para'] ?? 'todos',
                isset($_POST['publicado']) ? 1 : 0,
                $_POST['artigo_id']
            ]);
            $sucesso = 'Artigo atualizado com sucesso!';
        } elseif ($acao === 'excluir') {
            $stmt = $pdo->prepare("DELETE FROM artigos_ajuda WHERE id = ?");
            $stmt->execute([$_POST['artigo_id']]);
            $sucesso = 'Artigo excluído com sucesso!';
        }
    } catch (PDOException $e) {
        $erro = 'Erro: ' . $e->getMessage();
    }
}

// Buscar todos os artigos
try {
    $stmt = $pdo->query("SELECT * FROM artigos_ajuda ORDER BY ordem, id DESC");
    $artigos = $stmt->fetchAll();
} catch (PDOException $e) {
    $artigos = [];
    $erro = "⚠️ Execute a migration: migration_whitelabel_v2.sql";
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Artigos de Ajuda - Admin</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="assets/dashboard-style.css">
    <style><?php include 'sistema_theme.php'; ?></style>
    <style>
        body { background: var(--cor-fundo); padding: 20px; }
        .container { max-width: 1400px; margin: 0 auto; }
        .header { background: var(--cor-card); padding: 30px; border-radius: 15px; margin-bottom: 30px; border: 1px solid var(--cor-borda); }
        .header-title { font-size: 36px; color: var(--cor-primaria); margin-bottom: 10px; }
        .btn-new { background: var(--cor-primaria); color: #000; padding: 12px 25px; border-radius: 8px; text-decoration: none; font-weight: 700; display: inline-flex; align-items: center; gap: 10px; }
        .artigos-grid { display: grid; gap: 20px; }
        .artigo-card { background: var(--cor-card); padding: 25px; border-radius: 12px; border: 1px solid var(--cor-borda); display: flex; justify-content: space-between; align-items: start; }
        .artigo-info h3 { color: var(--cor-texto); margin-bottom: 8px; }
        .artigo-meta { display: flex; gap: 15px; margin-top: 10px; }
        .badge { padding: 5px 12px; border-radius: 12px; font-size: 12px; font-weight: 600; }
        .badge-publicado { background: var(--cor-sucesso); color: #fff; }
        .badge-rascunho { background: var(--cor-texto-secundario); color: #fff; }
        .artigo-acoes { display: flex; gap: 10px; }
        .btn-icon { padding: 10px 15px; border-radius: 8px; border: none; cursor: pointer; font-size: 14px; }
        .btn-edit { background: var(--cor-info); color: #fff; }
        .btn-delete { background: var(--cor-erro); color: #fff; }
        .modal { display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.8); z-index: 1000; }
        .modal.active { display: flex; align-items: center; justify-content: center; }
        .modal-content { background: var(--cor-card); padding: 30px; border-radius: 15px; max-width: 800px; width: 90%; max-height: 90vh; overflow-y: auto; }
        .form-group { margin-bottom: 20px; }
        .form-label { color: var(--cor-texto); font-weight: 600; margin-bottom: 8px; display: block; }
        .form-input, .form-textarea, .form-select { width: 100%; padding: 12px; background: var(--cor-secundaria); border: 2px solid var(--cor-borda); border-radius: 8px; color: var(--cor-texto); }
        .form-textarea { min-height: 300px; font-family: inherit; }
        .form-checkbox { display: flex; align-items: center; gap: 10px; }
        .btn-save { background: var(--cor-primaria); color: #000; padding: 12px 30px; border: none; border-radius: 8px; font-weight: 700; cursor: pointer; }
        .btn-cancel { background: var(--cor-secundaria); color: var(--cor-texto); padding: 12px 30px; border: 2px solid var(--cor-borda); border-radius: 8px; font-weight: 700; cursor: pointer; margin-left: 10px; }
    </style>
</head>
<body>
    <?php include 'assets/navbar.php'; ?>

    <div class="container">
        <div class="header">
            <h1 class="header-title"><i class="fas fa-book"></i> Artigos de Ajuda</h1>
            <p style="color: var(--cor-texto-secundario); margin-bottom: 20px;">Gerencie os tutoriais e artigos que aparecem na seção Ajuda para os usuários</p>
            <button class="btn-new" onclick="abrirModal()"><i class="fas fa-plus"></i> Novo Artigo</button>
        </div>

        <?php if ($sucesso): ?>
            <div class="alert alert-success" style="background: var(--cor-sucesso); color: #fff; padding: 15px; border-radius: 8px; margin-bottom: 20px;">
                <i class="fas fa-check-circle"></i> <?= $sucesso ?>
            </div>
        <?php endif; ?>

        <?php if ($erro): ?>
            <div class="alert alert-error" style="background: var(--cor-erro); color: #fff; padding: 15px; border-radius: 8px; margin-bottom: 20px;">
                <i class="fas fa-exclamation-triangle"></i> <?= $erro ?>
            </div>
        <?php endif; ?>

        <div class="artigos-grid">
            <?php foreach ($artigos as $artigo): ?>
                <div class="artigo-card">
                    <div class="artigo-info">
                        <h3><?= htmlspecialchars($artigo['titulo']) ?></h3>
                        <div class="artigo-meta">
                            <span class="badge <?= $artigo['publicado'] ? 'badge-publicado' : 'badge-rascunho' ?>">
                                <?= $artigo['publicado'] ? 'Publicado' : 'Rascunho' ?>
                            </span>
                            <span style="color: var(--cor-texto-secundario); font-size: 13px;">
                                <i class="fas fa-users"></i> <?= ucfirst($artigo['visivel_para']) ?>
                            </span>
                            <span style="color: var(--cor-texto-secundario); font-size: 13px;">
                                <i class="fas fa-sort-numeric-down"></i> Ordem: <?= $artigo['ordem'] ?>
                            </span>
                        </div>
                    </div>
                    <div class="artigo-acoes">
                        <button class="btn-icon btn-edit" onclick="editarArtigo(<?= htmlspecialchars(json_encode($artigo), ENT_QUOTES) ?>)">
                            <i class="fas fa-edit"></i> Editar
                        </button>
                        <form method="POST" style="display: inline;" onsubmit="return confirm('Deseja realmente excluir este artigo?');">
                            <input type="hidden" name="acao" value="excluir">
                            <input type="hidden" name="artigo_id" value="<?= $artigo['id'] ?>">
                            <button type="submit" class="btn-icon btn-delete"><i class="fas fa-trash"></i> Excluir</button>
                        </form>
                    </div>
                </div>
            <?php endforeach; ?>

            <?php if (empty($artigos)): ?>
                <div style="text-align: center; padding: 60px; color: var(--cor-texto-secundario);">
                    <i class="fas fa-book-open" style="font-size: 64px; margin-bottom: 20px; opacity: 0.3;"></i>
                    <p>Nenhum artigo criado ainda. Clique em "Novo Artigo" para começar.</p>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Modal Criar/Editar -->
    <div id="modalArtigo" class="modal">
        <div class="modal-content">
            <h2 id="modalTitulo" style="color: var(--cor-primaria); margin-bottom: 25px;">Novo Artigo</h2>
            <form method="POST" id="formArtigo">
                <input type="hidden" name="acao" id="acao" value="criar">
                <input type="hidden" name="artigo_id" id="artigo_id">

                <div class="form-group">
                    <label class="form-label">Título do Artigo</label>
                    <input type="text" name="titulo" id="titulo" class="form-input" required>
                </div>

                <div class="form-group">
                    <label class="form-label">Conteúdo (HTML permitido)</label>
                    <textarea name="conteudo" id="conteudo" class="form-textarea" required></textarea>
                    <small style="color: var(--cor-texto-secundario);">Você pode usar HTML: &lt;h3&gt;, &lt;p&gt;, &lt;strong&gt;, &lt;ul&gt;, &lt;ol&gt;, &lt;li&gt;</small>
                </div>

                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px;">
                    <div class="form-group">
                        <label class="form-label">Ordem de Exibição</label>
                        <input type="number" name="ordem" id="ordem" class="form-input" value="0" min="0">
                    </div>

                    <div class="form-group">
                        <label class="form-label">Visível Para</label>
                        <select name="visivel_para" id="visivel_para" class="form-select">
                            <option value="todos">Todos os Usuários</option>
                            <option value="promoter">Apenas Promoters</option>
                            <option value="produtor">Apenas Produtores</option>
                        </select>
                    </div>
                </div>

                <div class="form-group form-checkbox">
                    <input type="checkbox" name="publicado" id="publicado" checked>
                    <label for="publicado" style="margin: 0;">Publicar artigo (visível para usuários)</label>
                </div>

                <div style="margin-top: 30px;">
                    <button type="submit" class="btn-save"><i class="fas fa-save"></i> Salvar Artigo</button>
                    <button type="button" class="btn-cancel" onclick="fecharModal()"><i class="fas fa-times"></i> Cancelar</button>
                </div>
            </form>
        </div>
    </div>

    <script>
        function abrirModal() {
            document.getElementById('modalArtigo').classList.add('active');
            document.getElementById('modalTitulo').textContent = 'Novo Artigo';
            document.getElementById('formArtigo').reset();
            document.getElementById('acao').value = 'criar';
            document.getElementById('publicado').checked = true;
        }

        function editarArtigo(artigo) {
            document.getElementById('modalArtigo').classList.add('active');
            document.getElementById('modalTitulo').textContent = 'Editar Artigo';
            document.getElementById('acao').value = 'editar';
            document.getElementById('artigo_id').value = artigo.id;
            document.getElementById('titulo').value = artigo.titulo;
            document.getElementById('conteudo').value = artigo.conteudo;
            document.getElementById('ordem').value = artigo.ordem;
            document.getElementById('visivel_para').value = artigo.visivel_para;
            document.getElementById('publicado').checked = artigo.publicado == 1;
        }

        function fecharModal() {
            document.getElementById('modalArtigo').classList.remove('active');
        }

        // Fechar modal ao clicar fora
        document.getElementById('modalArtigo').addEventListener('click', function(e) {
            if (e.target === this) fecharModal();
        });
    </script>
</body>
</html>
