<?php
// admin_checkin.php - VALIDAÇÃO MELHORADA DE EVENTOS + FEEDBACK SONORO/VISUAL
require_once 'config.php';
require_once 'functions.php';
verificarRole(['admin']);

// Verificar se há um evento aberto
$evento_aberto = null;
if (isset($_SESSION['evento_checkin_aberto'])) {
    $evento_id = intval($_SESSION['evento_checkin_aberto']);
    $stmt = $pdo->prepare("SELECT * FROM eventos WHERE id = ?");
    $stmt->execute([$evento_id]);
    $evento_aberto = $stmt->fetch();
    
    // Se o evento não existe mais, limpar sessão
    if (!$evento_aberto) {
        unset($_SESSION['evento_checkin_aberto']);
    }
}

// Abrir evento para check-in
if (isset($_POST['abrir_evento'])) {
    $evento_id = intval($_POST['evento_id']);
    $_SESSION['evento_checkin_aberto'] = $evento_id;
    header('Location: admin_checkin.php');
    exit;
}

// Fechar evento
if (isset($_GET['fechar_evento'])) {
    unset($_SESSION['evento_checkin_aberto']);
    header('Location: admin_checkin.php');
    exit;
}

// Processar check-in via POST (AJAX)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['qr_code'])) {
    header('Content-Type: application/json');
    
    $qr_code = $_POST['qr_code'];
    
    // VALIDAÇÃO 1: Verificar se há evento aberto
    if (!isset($_SESSION['evento_checkin_aberto'])) {
        echo json_encode([
            'success' => false,
            'tipo_erro' => 'sem_evento',
            'message' => '<i class="fas fa-exclamation-triangle"></i> Nenhum evento aberto para check-in!',
            'detalhes' => 'Abra um evento antes de escanear os QR Codes.'
        ]);
        exit;
    }
    
    $evento_aberto_id = intval($_SESSION['evento_checkin_aberto']);
    
    // Buscar informações do evento aberto
    $stmt = $pdo->prepare("SELECT nome FROM eventos WHERE id = ?");
    $stmt->execute([$evento_aberto_id]);
    $evento_aberto_info = $stmt->fetch();
    
    // Buscar convidado
    $stmt = $pdo->prepare("
        SELECT c.*, e.id as evento_id, e.nome as evento_nome, u.nome as promoter_nome
        FROM convidados c
        INNER JOIN eventos e ON c.evento_id = e.id
        INNER JOIN usuarios u ON c.promoter_id = u.id
        WHERE c.qr_code = ?
    ");
    $stmt->execute([$qr_code]);
    $convidado = $stmt->fetch();
    
    // VALIDAÇÃO 2: QR Code existe?
    if (!$convidado) {
        echo json_encode([
            'success' => false,
            'tipo_erro' => 'qr_invalido',
            'message' => '<i class="fas fa-times-circle"></i> QR Code inválido ou não encontrado',
            'detalhes' => 'Este QR Code não está cadastrado no sistema.'
        ]);
        exit;
    }
    
    // VALIDAÇÃO 3: QR Code é do evento aberto? (CRÍTICO!)
    if ($convidado['evento_id'] != $evento_aberto_id) {
        echo json_encode([
            'success' => false,
            'tipo_erro' => 'evento_diferente',
            'message' => '<i class="fas fa-ban"></i> QR CODE DE OUTRO EVENTO!',
            'detalhes' => 'Este QR Code NÃO pode ser usado aqui.',
            'data' => [
                'nome_convidado' => $convidado['nome'],
                'evento_do_qrcode' => $convidado['evento_nome'],
                'evento_aberto' => $evento_aberto_info['nome']
            ]
        ]);
        exit;
    }
    
    // VALIDAÇÃO 4: Já fez check-in?
    if ($convidado['checkin']) {
        echo json_encode([
            'success' => false,
            'tipo_erro' => 'ja_registrado',
            'message' => '<i class="fas fa-exclamation-triangle"></i> Check-in já realizado!',
            'detalhes' => 'Este convidado já fez check-in anteriormente.',
            'data' => [
                'nome' => $convidado['nome'],
                'evento_nome' => $convidado['evento_nome'],
                'promoter_nome' => $convidado['promoter_nome'],
                'checkin_data' => $convidado['checkin_data']
            ]
        ]);
        exit;
    }
    
    // <i class="fas fa-check-circle"></i> TUDO OK - Realizar check-in
    $admin_id = $_SESSION['user_id'];
    $stmt = $pdo->prepare("UPDATE convidados SET checkin = 1, checkin_data = NOW(), checkin_by_admin_id = ? WHERE id = ?");
    $stmt->execute([$admin_id, $convidado['id']]);

    // Atualizar comissões automaticamente
    atualizarComissoes($pdo, $convidado['promoter_id'], $convidado['evento_id']);
    
    echo json_encode([
        'success' => true,
        'message' => '<i class="fas fa-check-circle"></i> Check-in realizado com sucesso!',
        'data' => [
            'nome' => $convidado['nome'],
            'evento_nome' => $convidado['evento_nome'],
            'promoter_nome' => $convidado['promoter_nome']
        ]
    ]);
    exit;
}

// Buscar eventos disponíveis para check-in
$stmt = $pdo->query("
    SELECT e.*, 
    (SELECT COUNT(*) FROM convidados c WHERE c.evento_id = e.id) as total_convidados,
    (SELECT COUNT(*) FROM convidados c WHERE c.evento_id = e.id AND c.checkin = 1) as total_checkins
    FROM eventos e 
    WHERE e.status = 'ativo'
    ORDER BY e.data DESC
");
$eventos_disponiveis = $stmt->fetchAll();

// Buscar estatísticas do evento aberto
$checkins_hoje = 0;
if ($evento_aberto) {
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as total 
        FROM convidados 
        WHERE evento_id = ? AND checkin = 1 AND DATE(checkin_data) = CURDATE()
    ");
    $stmt->execute([$evento_aberto['id']]);
    $checkins_hoje = $stmt->fetch()['total'];
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Check-in Scanner - Admin</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="assets/dashboard-style.css">
    <script src="https://cdnjs.cloudflare.com/ajax/libs/html5-qrcode/2.3.8/html5-qrcode.min.js"></script>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: #000;
            color: #fff;
            min-height: 100vh;
        }
        
        .navbar {
            background: #1a1a1a;
            border-bottom: 2px solid #333;
            padding: 15px 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .navbar-brand {
            font-size: 24px;
            font-weight: bold;
            color: #FFD700;
        }
        
        .nav-link {
            color: #fff;
            text-decoration: none;
            padding: 8px 15px;
            border-radius: 8px;
            margin: 0 5px;
            background: #2a2a2a;
            transition: all 0.3s;
        }
        
        .nav-link:hover {
            background: #FFD700;
            color: #000;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }
        
        .evento-selecionado {
            background: linear-gradient(135deg, #FFD700 0%, #FFA500 100%);
            border-radius: 15px;
            padding: 25px;
            margin-bottom: 30px;
            color: #000;
            box-shadow: 0 10px 30px rgba(255, 215, 0, 0.3);
        }
        
        .evento-selecionado h2 {
            font-size: 28px;
            margin-bottom: 10px;
        }
        
        .evento-info {
            display: flex;
            gap: 30px;
            margin-top: 15px;
            flex-wrap: wrap;
        }
        
        .evento-stat {
            display: flex;
            flex-direction: column;
        }
        
        .evento-stat-label {
            font-size: 12px;
            opacity: 0.8;
            text-transform: uppercase;
        }
        
        .evento-stat-value {
            font-size: 24px;
            font-weight: bold;
        }
        
        .btn-fechar {
            background: #ff4444;
            color: #fff;
            padding: 12px 24px;
            border: none;
            border-radius: 8px;
            font-weight: bold;
            cursor: pointer;
            text-decoration: none;
            display: inline-block;
            margin-top: 15px;
            transition: all 0.3s;
        }
        
        .btn-fechar:hover {
            background: #cc0000;
            transform: translateY(-2px);
        }
        
        #qr-reader {
            border-radius: 15px;
            overflow: hidden;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.5);
            margin-bottom: 30px;
        }
        
        #result-container {
            margin-top: 30px;
        }
        
        .result {
            border-radius: 15px;
            padding: 30px;
            text-align: center;
            animation: slideIn 0.3s ease-out;
        }
        
        @keyframes slideIn {
            from {
                opacity: 0;
                transform: translateY(-20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .result.success {
            background: linear-gradient(135deg, #00C851 0%, #007E33 100%);
            border: 3px solid #00C851;
        }
        
        .result.error {
            background: linear-gradient(135deg, #ff4444 0%, #cc0000 100%);
            border: 3px solid #ff4444;
        }
        
        .result.error.evento-diferente {
            background: linear-gradient(135deg, #ff6600 0%, #cc0000 100%);
            border: 4px solid #ff6600;
            animation: shake 0.5s;
        }
        
        @keyframes shake {
            0%, 100% { transform: translateX(0); }
            10%, 30%, 50%, 70%, 90% { transform: translateX(-10px); }
            20%, 40%, 60%, 80% { transform: translateX(10px); }
        }
        
        .result-icon {
            font-size: 80px;
            margin-bottom: 20px;
        }
        
        .result h2 {
            font-size: 32px;
            margin-bottom: 15px;
        }
        
        .result p {
            font-size: 18px;
            margin: 10px 0;
        }
        
        .info-box {
            background: rgba(0, 0, 0, 0.3);
            border-radius: 10px;
            padding: 20px;
            margin-top: 20px;
        }
        
        .info-item {
            display: flex;
            justify-content: space-between;
            padding: 12px 0;
            border-bottom: 1px solid rgba(255, 255, 255, 0.2);
        }
        
        .info-item:last-child {
            border-bottom: none;
        }
        
        .label {
            font-weight: 600;
            opacity: 0.9;
        }
        
        .value {
            font-weight: bold;
            text-align: right;
        }
        
        .alerta-evento {
            background: #fff;
            color: #ff0000;
            padding: 15px;
            border-radius: 10px;
            margin-top: 20px;
            font-weight: bold;
            font-size: 18px;
            border: 3px dashed #ff0000;
        }
        
        .selecionar-evento {
            background: #1a1a1a;
            border-radius: 15px;
            padding: 30px;
            margin-bottom: 30px;
        }
        
        .selecionar-evento h2 {
            color: #FFD700;
            margin-bottom: 20px;
        }
        
        .eventos-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 20px;
        }
        
        .evento-card {
            background: #2a2a2a;
            border: 2px solid #333;
            border-radius: 12px;
            padding: 20px;
            transition: all 0.3s;
            cursor: pointer;
        }
        
        .evento-card:hover {
            border-color: #FFD700;
            transform: translateY(-5px);
            box-shadow: 0 10px 30px rgba(255, 215, 0, 0.2);
        }
        
        .evento-card h3 {
            color: #FFD700;
            margin-bottom: 15px;
            font-size: 20px;
        }
        
        .evento-card-info {
            color: #999;
            font-size: 14px;
            margin: 8px 0;
        }
        
        .btn-abrir {
            background: #FFD700;
            color: #000;
            padding: 12px 24px;
            border: none;
            border-radius: 8px;
            font-weight: bold;
            cursor: pointer;
            width: 100%;
            margin-top: 15px;
            transition: all 0.3s;
        }
        
        .btn-abrir:hover {
            background: #FFA500;
            transform: translateY(-2px);
        }

        /* ============================================
           NOVO: TELAS FULL-SCREEN DE FEEDBACK
           ============================================ */
        .feedback-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            z-index: 9999;
            display: none;
            align-items: center;
            justify-content: center;
            animation: fadeIn 0.3s ease-out;
        }

        .feedback-overlay.show {
            display: flex;
        }

        .feedback-overlay.success {
            background: linear-gradient(135deg, #00C851 0%, #007E33 100%);
        }

        .feedback-overlay.error {
            background: linear-gradient(135deg, #ff4444 0%, #cc0000 100%);
        }

        .feedback-content {
            text-align: center;
            color: #fff;
            padding: 40px;
        }

        .feedback-icon {
            font-size: 120px;
            margin-bottom: 20px;
            animation: scaleIn 0.4s ease-out;
        }

        @keyframes scaleIn {
            from {
                transform: scale(0);
            }
            to {
                transform: scale(1);
            }
        }

        @keyframes fadeIn {
            from {
                opacity: 0;
            }
            to {
                opacity: 1;
            }
        }
    </style>
</head>
<body>
    <!-- NOVO: Telas de Feedback Full-Screen -->
    <div id="feedback-overlay" class="feedback-overlay">
        <div class="feedback-content">
            <div id="feedback-icon" class="feedback-icon"></div>
            <h2 id="feedback-message" style="font-size: 36px; margin-bottom: 20px;"></h2>
            <p id="feedback-details" style="font-size: 20px;"></p>
            <div id="feedback-info"></div>
        </div>
    </div>

    <nav class="navbar">
        <div class="navbar-brand"><i class="fas fa-qrcode"></i> Check-in Scanner</div>
        <div>
            <a href="dashboard.php" class="nav-link">← Dashboard</a>
            <a href="logout.php" class="nav-link">Sair</a>
        </div>
    </nav>
    
    <div class="container">
        <?php if ($evento_aberto): ?>
            <!-- Evento Aberto -->
            <div class="evento-selecionado">
                <h2><i class="fas fa-theater-masks"></i> <?= htmlspecialchars($evento_aberto['nome']) ?></h2>
                <p>Evento aberto para check-in</p>
                
                <div class="evento-info">
                    <div class="evento-stat">
                        <span class="evento-stat-label">Check-ins hoje</span>
                        <span class="evento-stat-value" id="total-checkins"><?= $checkins_hoje ?></span>
                    </div>
                    <div class="evento-stat">
                        <span class="evento-stat-label">Último check-in</span>
                        <span class="evento-stat-value" id="ultimo-checkin">--:--</span>
                    </div>
                    <div class="evento-stat">
                        <span class="evento-stat-label"><i class="fas fa-calendar-alt"></i> Data</span>
                        <span class="evento-stat-value"><?= formatarData($evento_aberto['data']) ?></span>
                    </div>
                </div>
                
                <a href="?fechar_evento" class="btn-fechar"><i class="fas fa-lock"></i> Fechar Evento</a>
            </div>
            
            <!-- Scanner QR Code -->
            <div id="qr-reader" style="width: 100%; max-width: 600px; margin: 0 auto;"></div>
            
            <!-- Resultado do Scan (mantido para fallback) -->
            <div id="result-container"></div>
            
        <?php else: ?>
            <!-- Selecionar Evento -->
            <div class="selecionar-evento">
                <h2>Selecione um Evento para Iniciar o Check-in</h2>
                <p style="color: #999; margin-bottom: 30px;">Escolha qual evento você deseja abrir para fazer check-in dos convidados</p>
                
                <?php if (count($eventos_disponiveis) > 0): ?>
                    <div class="eventos-grid">
                        <?php foreach ($eventos_disponiveis as $evento): ?>
                            <div class="evento-card">
                                <h3><?= htmlspecialchars($evento['nome']) ?></h3>
                                <div class="evento-card-info"><i class="fas fa-calendar-alt"></i> <?= formatarData($evento['data']) ?></div>
                                <div class="evento-card-info"><i class="fas fa-users"></i> <?= $evento['total_convidados'] ?> convidados</div>
                                <div class="evento-card-info"><i class="fas fa-check-circle"></i> <?= $evento['total_checkins'] ?> check-ins</div>
                                
                                <form method="POST" style="margin: 0;">
                                    <input type="hidden" name="evento_id" value="<?= $evento['id'] ?>">
                                    <button type="submit" name="abrir_evento" class="btn-abrir">
                                        <i class="fas fa-rocket"></i> Abrir para Check-in
                                    </button>
                                </form>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php else: ?>
                    <p style="text-align: center; padding: 60px; color: #666;">
                        Nenhum evento disponível
                    </p>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>
    
    <?php if ($evento_aberto): ?>
    <script>
        let html5QrCode;
        let scannerAtivo = true;
        let totalCheckins = <?= $checkins_hoje ?>;
        
        // ============================================
        // NOVO: SISTEMA DE SONS
        // ============================================
        const audioContext = new (window.AudioContext || window.webkitAudioContext)();
        
        function tocarSomSucesso() {
            // BIP de sucesso (frequência alta e curta)
            const oscillator = audioContext.createOscillator();
            const gainNode = audioContext.createGain();
            
            oscillator.connect(gainNode);
            gainNode.connect(audioContext.destination);
            
            oscillator.frequency.value = 800; // Frequência alta
            oscillator.type = 'sine';
            
            gainNode.gain.setValueAtTime(0.3, audioContext.currentTime);
            gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.2);
            
            oscillator.start(audioContext.currentTime);
            oscillator.stop(audioContext.currentTime + 0.2);
        }
        
        function tocarSomErro() {
            // Som de erro (frequência baixa e longa)
            const oscillator = audioContext.createOscillator();
            const gainNode = audioContext.createGain();
            
            oscillator.connect(gainNode);
            gainNode.connect(audioContext.destination);
            
            oscillator.frequency.value = 200; // Frequência baixa
            oscillator.type = 'sawtooth';
            
            gainNode.gain.setValueAtTime(0.3, audioContext.currentTime);
            gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.5);
            
            oscillator.start(audioContext.currentTime);
            oscillator.stop(audioContext.currentTime + 0.5);
        }
        
        // ============================================
        // NOVO: FUNÇÃO DE VIBRAÇÃO
        // ============================================
        function vibrar() {
            if ('vibrate' in navigator) {
                // Vibração padrão: 500ms
                navigator.vibrate(500);
            }
        }
        
        // ============================================
        // NOVO: MOSTRAR FEEDBACK FULL-SCREEN
        // ============================================
        function mostrarFeedbackFullScreen(data) {
            const overlay = document.getElementById('feedback-overlay');
            const icon = document.getElementById('feedback-icon');
            const message = document.getElementById('feedback-message');
            const details = document.getElementById('feedback-details');
            const info = document.getElementById('feedback-info');
            
            // Limpar classes anteriores
            overlay.className = 'feedback-overlay';
            
            if (data.success) {
                // SUCESSO
                overlay.classList.add('success', 'show');
                icon.innerHTML = '<i class="fas fa-check-circle"></i>';
                message.textContent = 'Check-in Realizado!';
                details.textContent = data.data.nome;
                
                info.innerHTML = `
                    <div style="margin-top: 30px; font-size: 18px;">
                        <p><strong>Evento:</strong> ${data.data.evento_nome}</p>
                        <p style="margin-top: 10px;"><strong>Promoter:</strong> ${data.data.promoter_nome}</p>
                        <p style="margin-top: 20px;"><i class="fas fa-money-bill-wave"></i> Comissão atualizada automaticamente!</p>
                    </div>
                `;
                
                // Tocar som de sucesso
                tocarSomSucesso();
                
            } else {
                // ERRO
                overlay.classList.add('error', 'show');

                // Vibrar em caso de erro
                vibrar();

                // Tocar som de erro
                tocarSomErro();

                // Definir ícone e mensagem baseado no tipo de erro
                if (data.tipo_erro === 'evento_diferente') {
                    icon.innerHTML = '<i class="fas fa-ban"></i>';
                    message.textContent = 'QR CODE DE OUTRO EVENTO!';
                    details.textContent = 'Este QR Code não pode ser usado aqui';
                    
                    info.innerHTML = `
                        <div style="margin-top: 30px; font-size: 18px; background: rgba(0,0,0,0.3); padding: 20px; border-radius: 10px;">
                            <p><strong>Convidado:</strong> ${data.data.nome_convidado}</p>
                            <p style="margin-top: 10px;"><strong>QR Code pertence ao:</strong><br>${data.data.evento_do_qrcode}</p>
                            <p style="margin-top: 10px;"><strong>Evento aberto:</strong><br>${data.data.evento_aberto}</p>
                        </div>
                    `;
                } else if (data.tipo_erro === 'ja_registrado') {
                    icon.innerHTML = '<i class="fas fa-exclamation-triangle"></i>';
                    message.textContent = 'Check-in Já Realizado!';
                    details.textContent = data.data.nome;
                    
                    info.innerHTML = `
                        <div style="margin-top: 30px; font-size: 18px;">
                            <p><strong>Evento:</strong> ${data.data.evento_nome}</p>
                            <p style="margin-top: 10px;"><strong>Check-in em:</strong><br>${new Date(data.data.checkin_data).toLocaleString('pt-BR')}</p>
                        </div>
                    `;
                } else if (data.tipo_erro === 'qr_invalido') {
                    icon.innerHTML = '<i class="fas fa-times-circle"></i>';
                    message.textContent = 'QR Code Inválido';
                    details.textContent = 'Este QR Code não está cadastrado no sistema';
                    info.innerHTML = '';
                } else {
                    icon.innerHTML = '<i class="fas fa-exclamation-triangle"></i>';
                    message.textContent = data.message;
                    details.textContent = data.detalhes || '';
                    info.innerHTML = '';
                }
            }
            
            // Ocultar após 4 segundos (conforme especificação)
            setTimeout(() => {
                overlay.classList.remove('show');
            }, 4000);
        }
        
        function onScanSuccess(decodedText, decodedResult) {
            if (!scannerAtivo) return;
            
            scannerAtivo = false;
            html5QrCode.pause();
            
            // Enviar para o servidor
            fetch('admin_checkin.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'qr_code=' + encodeURIComponent(decodedText)
            })
            .then(response => response.json())
            .then(data => {
                // NOVO: Mostrar feedback full-screen
                mostrarFeedbackFullScreen(data);
                
                // Atualizar estatísticas se sucesso
                if (data.success) {
                    totalCheckins++;
                    document.getElementById('total-checkins').textContent = totalCheckins;
                    document.getElementById('ultimo-checkin').textContent = new Date().toLocaleTimeString('pt-BR', {hour: '2-digit', minute: '2-digit'});
                }
                
                // Aguardar 4 segundos antes de continuar (conforme especificação)
                setTimeout(() => {
                    scannerAtivo = true;
                    html5QrCode.resume();
                }, 4000);
            })
            .catch(error => {
                console.error('Erro:', error);
                scannerAtivo = true;
                html5QrCode.resume();
            });
        }
        
        // Iniciar scanner
        html5QrCode = new Html5Qrcode("qr-reader");
        html5QrCode.start(
            { facingMode: "environment" },
            {
                fps: 10,
                qrbox: { width: 250, height: 250 }
            },
            onScanSuccess
        ).catch(err => {
            console.error("Erro ao iniciar scanner:", err);
            document.getElementById('qr-reader').innerHTML = `
                <div style="padding: 40px; text-align: center; color: #ff4444;">
                    <h3><i class="fas fa-exclamation-triangle"></i> Erro ao acessar a câmera</h3>
                    <p style="margin-top: 15px;">Verifique se você deu permissão para acessar a câmera.</p>
                </div>
            `;
        });
    </script>
    <?php endif; ?>
</body>
</html>