<?php
// admin_convidados.php - Visualizar todos os convidados por evento
require_once 'config.php';
verificarRole(['admin']);

// Buscar todos os eventos
$stmt = $pdo->query("
    SELECT e.*, 
    (SELECT COUNT(*) FROM convidados c WHERE c.evento_id = e.id) as total_convidados,
    (SELECT COUNT(*) FROM convidados c WHERE c.evento_id = e.id AND c.checkin = 1) as total_checkins,
    (SELECT COUNT(DISTINCT c.promoter_id) FROM convidados c WHERE c.evento_id = e.id) as total_promoters
    FROM eventos e 
    ORDER BY e.data DESC
");
$eventos = $stmt->fetchAll();

// Se um evento foi selecionado
$evento_selecionado = null;
$convidados = [];
$filtro = isset($_GET['filtro']) ? $_GET['filtro'] : 'todos';

if (isset($_GET['evento_id'])) {
    $evento_id = intval($_GET['evento_id']);
    
    $stmt = $pdo->prepare("SELECT * FROM eventos WHERE id = ?");
    $stmt->execute([$evento_id]);
    $evento_selecionado = $stmt->fetch();
    
    if ($evento_selecionado) {
        // Buscar convidados conforme filtro
        $where = "c.evento_id = ?";
        $params = [$evento_id];
        
        if ($filtro === 'checkin') {
            $where .= " AND c.checkin = 1";
        } elseif ($filtro === 'pendente') {
            $where .= " AND c.checkin = 0";
        }
        
        $stmt = $pdo->prepare("
            SELECT c.*, u.nome as promoter_nome, u.telefone as promoter_telefone, u.instagram as promoter_instagram
            FROM convidados c
            INNER JOIN usuarios u ON c.promoter_id = u.id
            WHERE $where
            ORDER BY c.created_at DESC
        ");
        $stmt->execute($params);
        $convidados = $stmt->fetchAll();
    }
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Convidados - Admin</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="assets/dashboard-style.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root[data-theme="dark"] {
            --bg-primary: #000000;
            --bg-secondary: #1a1a1a;
            --bg-card: #2a2a2a;
            --color-primary: #FFD700;
            --color-secondary: #FFA500;
            --color-text: #ffffff;
            --color-text-secondary: #999999;
            --border-color: #333333;
        }
        
        :root[data-theme="light"] {
            --bg-primary: #ffffff;
            --bg-secondary: #f5f5f5;
            --bg-card: #ffffff;
            --color-primary: #003366;
            --color-secondary: #0066cc;
            --color-text: #000000;
            --color-text-secondary: #666666;
            --border-color: #e0e0e0;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: var(--bg-secondary);
            color: var(--color-text);
        }
        
        .navbar {
            background: var(--bg-primary);
            border-bottom: 2px solid var(--border-color);
            padding: 15px 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            position: sticky;
            top: 0;
            z-index: 100;
        }
        
        .navbar-brand {
            font-size: 24px;
            font-weight: bold;
            color: var(--color-primary);
        }
        
        .nav-link {
            color: var(--color-text);
            text-decoration: none;
            padding: 8px 15px;
            border-radius: 8px;
            margin: 0 5px;
        }
        
        .nav-link:hover {
            background: var(--bg-card);
            color: var(--color-primary);
        }
        
        .container {
            max-width: 1600px;
            margin: 0 auto;
            padding: 30px 20px;
        }
        
        .eventos-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .evento-card {
            background: var(--bg-card);
            border: 2px solid var(--border-color);
            border-radius: 15px;
            padding: 20px;
            cursor: pointer;
            transition: all 0.3s;
            text-decoration: none;
            color: var(--color-text);
        }
        
        .evento-card:hover, .evento-card.active {
            border-color: var(--color-primary);
            transform: translateY(-5px);
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
        }
        
        .evento-nome {
            font-size: 18px;
            font-weight: bold;
            color: var(--color-primary);
            margin-bottom: 10px;
        }
        
        .evento-stats {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 10px;
            margin-top: 15px;
        }
        
        .stat-mini {
            text-align: center;
            padding: 10px;
            background: var(--bg-primary);
            border-radius: 8px;
        }
        
        .stat-value {
            font-size: 20px;
            font-weight: bold;
            color: var(--color-primary);
        }
        
        .stat-label {
            font-size: 10px;
            color: var(--color-text-secondary);
        }
        
        .filtros {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
        }
        
        .filtro-btn {
            padding: 10px 20px;
            border: 2px solid var(--border-color);
            border-radius: 8px;
            background: var(--bg-card);
            color: var(--color-text);
            cursor: pointer;
            text-decoration: none;
            transition: all 0.3s;
        }
        
        .filtro-btn:hover, .filtro-btn.active {
            border-color: var(--color-primary);
            background: var(--bg-primary);
            color: var(--color-primary);
        }
        
        .search-box {
            margin-bottom: 20px;
        }
        
        .search-input {
            width: 100%;
            padding: 15px;
            border: 2px solid var(--border-color);
            border-radius: 10px;
            background: var(--bg-card);
            color: var(--color-text);
            font-size: 16px;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
            background: var(--bg-card);
            border-radius: 15px;
            overflow: hidden;
        }
        
        th {
            background: var(--bg-primary);
            padding: 15px;
            text-align: left;
            font-size: 13px;
            color: var(--color-primary);
            border-bottom: 2px solid var(--border-color);
        }
        
        td {
            padding: 15px;
            border-bottom: 1px solid var(--border-color);
            font-size: 13px;
        }
        
        tr:hover {
            background: var(--bg-primary);
        }
        
        .badge {
            display: inline-block;
            padding: 4px 10px;
            border-radius: 20px;
            font-size: 11px;
            font-weight: bold;
        }
        
        .badge-success {
            background: #00C851;
            color: white;
        }
        
        .badge-warning {
            background: #FFA500;
            color: white;
        }
        
        .btn-export {
            padding: 12px 25px;
            background: linear-gradient(135deg, var(--color-primary), var(--color-secondary));
            color: #000;
            border: none;
            border-radius: 10px;
            font-weight: bold;
            cursor: pointer;
            text-decoration: none;
            display: inline-block;
        }
        
        .btn-export:hover {
            opacity: 0.9;
        }
        
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: var(--color-text-secondary);
        }
        
        .empty-icon {
            font-size: 64px;
            margin-bottom: 20px;
        }
        
        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        
        @media (max-width: 768px) {
            .eventos-grid {
                grid-template-columns: 1fr;
            }
            
            table {
                font-size: 11px;
            }
            
            th, td {
                padding: 8px 5px;
            }
            
            .filtros {
                flex-wrap: wrap;
            }
        }
    </style>
</head>
<body>
    <nav class="navbar">
        <div class="navbar-brand"><i class="fas fa-users"></i> Convidados</div>
        <div>
            <a href="dashboard.php" class="nav-link">Dashboard</a>
            <a href="admin_eventos.php" class="nav-link">Eventos</a>
            <a href="admin_usuarios.php" class="nav-link">Usuários</a>
            <a href="admin_checkin_manual.php" class="nav-link">Check-in Manual</a>
            <a href="logout.php" class="nav-link">Sair</a>
        </div>
    </nav>
    
    <div class="container">
        <h1 style="font-size: 32px; color: var(--color-primary); margin-bottom: 30px;">
            <i class="fas fa-users"></i> Gerenciar Convidados
        </h1>
        
        <!-- Seletor de Eventos -->
        <div class="eventos-grid">
            <?php foreach ($eventos as $evt): ?>
                <a href="?evento_id=<?= $evt['id'] ?>" class="evento-card <?= (isset($_GET['evento_id']) && $_GET['evento_id'] == $evt['id']) ? 'active' : '' ?>">
                    <div class="evento-nome"><?= htmlspecialchars($evt['nome']) ?></div>
                    <div style="font-size: 13px; color: var(--color-text-secondary); margin-bottom: 10px;">
                        <i class="fas fa-calendar-alt"></i> <?= formatarData($evt['data']) ?>
                    </div>
                    <div class="evento-stats">
                        <div class="stat-mini">
                            <div class="stat-value"><?= $evt['total_convidados'] ?></div>
                            <div class="stat-label">Convidados</div>
                        </div>
                        <div class="stat-mini">
                            <div class="stat-value"><?= $evt['total_checkins'] ?></div>
                            <div class="stat-label">Check-ins</div>
                        </div>
                        <div class="stat-mini">
                            <div class="stat-value"><?= $evt['total_promoters'] ?></div>
                            <div class="stat-label">Promoters</div>
                        </div>
                    </div>
                </a>
            <?php endforeach; ?>
        </div>
        
        <?php if ($evento_selecionado): ?>
            <div class="card-header">
                <div>
                    <h2 style="color: var(--color-primary); margin-bottom: 5px;">
                        <?= htmlspecialchars($evento_selecionado['nome']) ?>
                    </h2>
                    <p style="color: var(--color-text-secondary);">
                        Total: <?= count($convidados) ?> convidados
                    </p>
                </div>
                <button class="btn-export" onclick="exportarCSV()"><i class="fas fa-file-excel"></i> Exportar CSV</button>
            </div>
            
            <!-- Filtros -->
            <div class="filtros">
                <a href="?evento_id=<?= $evento_selecionado['id'] ?>&filtro=todos" class="filtro-btn <?= $filtro === 'todos' ? 'active' : '' ?>">
                    📋 Todos
                </a>
                <a href="?evento_id=<?= $evento_selecionado['id'] ?>&filtro=checkin" class="filtro-btn <?= $filtro === 'checkin' ? 'active' : '' ?>">
                    ✅ Com Check-in
                </a>
                <a href="?evento_id=<?= $evento_selecionado['id'] ?>&filtro=pendente" class="filtro-btn <?= $filtro === 'pendente' ? 'active' : '' ?>">
                    ⏳ Pendentes
                </a>
            </div>
            
            <!-- Busca -->
            <div class="search-box">
                <input type="text" class="search-input" id="searchInput" placeholder="🔍 Buscar por nome, telefone, email ou promoter..." onkeyup="filtrarTabela()">
            </div>
            
            <!-- Tabela de Convidados -->
            <?php if (count($convidados) > 0): ?>
                <table id="tabelaConvidados">
                    <thead>
                        <tr>
                            <th>Nome</th>
                            <th>Telefone</th>
                            <th>Email</th>
                            <th>Promoter</th>
                            <th>Contato Promoter</th>
                            <th>Cadastro</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($convidados as $conv): ?>
                            <tr>
                                <td><strong><?= htmlspecialchars($conv['nome']) ?></strong></td>
                                <td>
                                    <a href="https://wa.me/<?= preg_replace('/[^0-9]/', '', $conv['whatsapp']) ?>" target="_blank" style="color: var(--color-primary);">
                                        <?= htmlspecialchars($conv['whatsapp']) ?>
                                    </a>
                                </td>
                                <td><?= htmlspecialchars($conv['email']) ?></td>
                                <td>
                                    <strong><?= htmlspecialchars($conv['promoter_nome']) ?></strong>
                                    <?php if ($conv['promoter_instagram']): ?>
                                        <br><small style="color: var(--color-text-secondary);">📷 <?= htmlspecialchars($conv['promoter_instagram']) ?></small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <a href="https://wa.me/<?= preg_replace('/[^0-9]/', '', $conv['promoter_telefone']) ?>" target="_blank" style="color: var(--color-primary);">
                                        <?= htmlspecialchars($conv['promoter_telefone']) ?>
                                    </a>
                                </td>
                                <td><?= formatarData($conv['created_at']) ?></td>
                                <td>
                                    <?php if ($conv['checkin']): ?>
                                        <span class="badge badge-success"><i class="fas fa-check"></i> Presente</span>
                                        <br><small style="color: var(--color-text-secondary);"><?= formatarDataHora($conv['checkin_data']) ?></small>
                                    <?php else: ?>
                                        <span class="badge badge-warning">⏳ Pendente</span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php else: ?>
                <div class="empty-state">
                    <div class="empty-icon">📭</div>
                    <h3>Nenhum convidado encontrado</h3>
                    <p>Altere o filtro ou selecione outro evento.</p>
                </div>
            <?php endif; ?>
        <?php elseif (count($eventos) === 0): ?>
            <div class="empty-state">
                <div class="empty-icon">🎪</div>
                <h3>Nenhum evento cadastrado</h3>
                <p>Crie um evento para começar.</p>
            </div>
        <?php endif; ?>
    </div>
    
    <script>
        const savedTheme = localStorage.getItem('theme') || 'dark';
        document.documentElement.setAttribute('data-theme', savedTheme);
        
        function filtrarTabela() {
            const searchTerm = document.getElementById('searchInput').value.toLowerCase();
            const table = document.getElementById('tabelaConvidados');
            const rows = table.getElementsByTagName('tr');
            
            for (let i = 1; i < rows.length; i++) {
                const row = rows[i];
                const text = row.textContent.toLowerCase();
                
                if (text.includes(searchTerm)) {
                    row.style.display = '';
                } else {
                    row.style.display = 'none';
                }
            }
        }
        
        function exportarCSV() {
            const table = document.getElementById('tabelaConvidados');
            let csv = [];
            
            // Cabeçalhos
            const headers = [];
            const headerCells = table.querySelectorAll('thead th');
            headerCells.forEach(cell => {
                headers.push(cell.textContent.trim());
            });
            csv.push(headers.join(';'));
            
            // Dados
            const rows = table.querySelectorAll('tbody tr');
            rows.forEach(row => {
                if (row.style.display !== 'none') {
                    const rowData = [];
                    const cells = row.querySelectorAll('td');
                    cells.forEach(cell => {
                        rowData.push(cell.textContent.trim().replace(/\n/g, ' '));
                    });
                    csv.push(rowData.join(';'));
                }
            });
            
            // Download
            const csvContent = csv.join('\n');
            const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
            const link = document.createElement('a');
            link.href = URL.createObjectURL(blob);
            link.download = 'convidados_<?= $evento_selecionado ? str_replace(' ', '_', $evento_selecionado['nome']) : 'evento' ?>_<?= date('Y-m-d') ?>.csv';
            link.click();
        }
    </script>
</body>
</html>