<?php
// admin_saques.php - CORRIGIDO PARA ERRO DE STATUS
require_once 'config.php';
verificarRole(['admin']);

$sucesso = '';
$erro = '';

// Aprovar saque
if (isset($_GET['aprovar'])) {
    try {
        $saque_id = intval($_GET['aprovar']);
        
        // CORREÇÃO: Usar 'processando' ao invés de 'aprovado'
        $stmt = $pdo->prepare("UPDATE saques SET status = 'processando' WHERE id = ?");
        if ($stmt->execute([$saque_id])) {
            $sucesso = '<i class="fas fa-check-circle"></i> Saque aprovado! Agora está na lista "A Pagar".';
        } else {
            $erro = 'Erro ao aprovar saque.';
        }
    } catch (Exception $e) {
        $erro = 'Erro: ' . $e->getMessage();
    }
}

// Rejeitar saque
if (isset($_GET['rejeitar'])) {
    try {
        $saque_id = intval($_GET['rejeitar']);
        
        // Buscar dados do saque
        $stmt = $pdo->prepare("SELECT user_id FROM saques WHERE id = ?");
        $stmt->execute([$saque_id]);
        $saque = $stmt->fetch();
        
        if ($saque) {
            // Marcar como cancelado
            $stmt = $pdo->prepare("UPDATE saques SET status = 'cancelado' WHERE id = ?");
            $stmt->execute([$saque_id]);
            
            // Devolver comissões para disponível
            $stmt = $pdo->prepare("UPDATE comissoes SET status = 'disponivel' WHERE user_id = ? AND status = 'solicitado'");
            $stmt->execute([$saque['user_id']]);
            
            $sucesso = '<i class="fas fa-times-circle"></i> Saque rejeitado. Saldo devolvido ao promoter.';
        }
    } catch (Exception $e) {
        $erro = 'Erro: ' . $e->getMessage();
    }
}

// Marcar como pago
if (isset($_GET['pagar'])) {
    try {
        $saque_id = intval($_GET['pagar']);

        // Buscar dados do saque
        $stmt = $pdo->prepare("SELECT user_id, valor, tipo FROM saques WHERE id = ?");
        $stmt->execute([$saque_id]);
        $saque = $stmt->fetch();

        if ($saque) {
            // CORREÇÃO: Usar 'concluido' (sem acento)
            $stmt = $pdo->prepare("UPDATE saques SET status = 'concluido', processado_at = NOW() WHERE id = ?");
            $stmt->execute([$saque_id]);

            // Se for cartão de consumo, descontar do crédito de consumação
            if ($saque['tipo'] === 'cartao_consumo') {
                // Marcar créditos de consumação como usados (descontar o valor)
                $valor_restante = $saque['valor'];

                $stmt = $pdo->prepare("
                    SELECT id, valor
                    FROM saques
                    WHERE user_id = ? AND tipo = 'credito' AND status IN ('processando', 'concluido')
                    ORDER BY created_at ASC
                ");
                $stmt->execute([$saque['user_id']]);
                $creditos = $stmt->fetchAll();

                foreach ($creditos as $credito) {
                    if ($valor_restante <= 0) break;

                    if ($credito['valor'] <= $valor_restante) {
                        // Marcar crédito inteiro como usado
                        $stmt = $pdo->prepare("UPDATE saques SET status = 'usado' WHERE id = ?");
                        $stmt->execute([$credito['id']]);
                        $valor_restante -= $credito['valor'];
                    } else {
                        // Reduzir parcialmente o crédito
                        $novo_valor = $credito['valor'] - $valor_restante;
                        $stmt = $pdo->prepare("UPDATE saques SET valor = ? WHERE id = ?");
                        $stmt->execute([$novo_valor, $credito['id']]);
                        $valor_restante = 0;
                    }
                }
            } else {
                // Para saques normais (PIX e crédito), marcar comissões como pagas
                $stmt = $pdo->prepare("UPDATE comissoes SET status = 'pago' WHERE user_id = ? AND status = 'solicitado'");
                $stmt->execute([$saque['user_id']]);
            }

            $sucesso = '<i class="fas fa-money-bill-wave"></i> Pagamento confirmado!';
        }
    } catch (Exception $e) {
        $erro = 'Erro: ' . $e->getMessage();
    }
}

// Buscar saques pendentes de aprovação
$stmt = $pdo->query("
    SELECT s.*, u.nome, u.telefone, u.email, u.pix as user_pix,
    (SELECT GROUP_CONCAT(DISTINCT e.nome SEPARATOR ', ') 
     FROM comissoes c 
     LEFT JOIN eventos e ON c.evento_id = e.id 
     WHERE c.user_id = s.user_id AND c.status = 'solicitado') as eventos
    FROM saques s
    INNER JOIN usuarios u ON s.user_id = u.id
    WHERE s.status = 'pendente'
    ORDER BY s.created_at DESC
");
$saques_pendentes = $stmt->fetchAll();

// Buscar saques aprovados (a pagar) - CORREÇÃO: buscar status 'processando'
$stmt = $pdo->query("
    SELECT s.*, u.nome, u.telefone, u.email, u.pix as user_pix,
    (SELECT GROUP_CONCAT(DISTINCT e.nome SEPARATOR ', ') 
     FROM comissoes c 
     LEFT JOIN eventos e ON c.evento_id = e.id 
     WHERE c.user_id = s.user_id AND c.status = 'solicitado') as eventos
    FROM saques s
    INNER JOIN usuarios u ON s.user_id = u.id
    WHERE s.status = 'processando'
    ORDER BY s.created_at DESC
");
$saques_aprovados = $stmt->fetchAll();

// Buscar histórico de pagos
$filtro_nome = isset($_GET['nome']) ? $_GET['nome'] : '';
$filtro_data = isset($_GET['data']) ? $_GET['data'] : '';

$where = "s.status = 'concluido'";
$params = [];

if ($filtro_nome) {
    $where .= " AND u.nome LIKE ?";
    $params[] = "%$filtro_nome%";
}

if ($filtro_data) {
    $where .= " AND DATE(s.processado_at) = ?";
    $params[] = $filtro_data;
}

$sql = "
    SELECT s.*, u.nome, u.telefone, u.email,
    (SELECT GROUP_CONCAT(DISTINCT e.nome SEPARATOR ', ') 
     FROM comissoes c 
     LEFT JOIN eventos e ON c.evento_id = e.id 
     WHERE c.user_id = s.user_id AND c.status = 'pago') as eventos
    FROM saques s
    INNER JOIN usuarios u ON s.user_id = u.id
    WHERE $where
    ORDER BY s.processado_at DESC
    LIMIT 100
";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$saques_pagos = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gerenciar Saques - Admin</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="assets/dashboard-style.css">
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        :root[data-theme="dark"] {
            --bg-primary: #000000;
            --bg-secondary: #1a1a1a;
            --bg-card: #2a2a2a;
            --color-primary: #FFD700;
            --color-text: #ffffff;
            --color-text-secondary: #999999;
            --border-color: #333333;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: var(--bg-secondary);
            color: var(--color-text);
        }
        
        .navbar {
            background: var(--bg-primary);
            border-bottom: 2px solid var(--border-color);
            padding: 15px 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            position: sticky;
            top: 0;
            z-index: 100;
        }
        
        .navbar-brand {
            font-size: 24px;
            font-weight: bold;
            color: var(--color-primary);
        }
        
        .navbar-menu {
            display: flex;
            gap: 10px;
            align-items: center;
        }
        
        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            text-decoration: none;
            display: inline-block;
            transition: all 0.3s;
        }
        
        .btn-primary {
            background: var(--color-primary);
            color: #000;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(255, 215, 0, 0.3);
        }
        
        .btn-success {
            background: #00C851;
            color: white;
        }
        
        .btn-success:hover {
            background: #00A041;
        }
        
        .btn-danger {
            background: #ff4444;
            color: white;
        }
        
        .btn-danger:hover {
            background: #cc0000;
        }
        
        .container {
            max-width: 1200px;
            margin: 20px auto;
            padding: 0 20px;
        }
        
        .alert {
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            font-weight: 500;
        }
        
        .alert-success {
            background: rgba(0, 200, 81, 0.2);
            border: 1px solid #00C851;
            color: #00C851;
        }
        
        .alert-error {
            background: rgba(255, 68, 68, 0.2);
            border: 1px solid #ff4444;
            color: #ff4444;
        }
        
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
        }
        
        .page-title {
            font-size: 32px;
            font-weight: bold;
            color: var(--color-primary);
        }
        
        .tabs {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
            border-bottom: 2px solid var(--border-color);
        }
        
        .tab {
            padding: 15px 30px;
            cursor: pointer;
            border: none;
            background: none;
            color: var(--color-text-secondary);
            font-size: 16px;
            font-weight: 600;
            border-bottom: 3px solid transparent;
            transition: all 0.3s;
        }
        
        .tab:hover {
            color: var(--color-primary);
        }
        
        .tab.active {
            color: var(--color-primary);
            border-bottom-color: var(--color-primary);
        }
        
        .tab-content {
            display: none;
        }
        
        .tab-content.active {
            display: block;
        }
        
        .saque-card {
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 15px;
        }
        
        .saque-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
            padding-bottom: 15px;
            border-bottom: 1px solid var(--border-color);
        }
        
        .promoter-nome {
            font-size: 20px;
            font-weight: bold;
            color: var(--color-text);
        }
        
        .saque-valor {
            font-size: 28px;
            font-weight: bold;
            color: var(--color-primary);
        }
        
        .saque-info {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 15px;
            margin-bottom: 15px;
        }
        
        .info-item {
            display: flex;
            flex-direction: column;
            gap: 5px;
        }
        
        .info-label {
            font-size: 12px;
            color: var(--color-text-secondary);
            text-transform: uppercase;
            font-weight: 600;
        }
        
        .info-value {
            font-size: 14px;
            color: var(--color-text);
        }
        
        .saque-actions {
            display: flex;
            gap: 10px;
            margin-top: 15px;
            padding-top: 15px;
            border-top: 1px solid var(--border-color);
        }
        
        .badge {
            display: inline-block;
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
        }
        
        .badge-pix {
            background: rgba(0, 200, 81, 0.2);
            color: #00C851;
            border: 1px solid #00C851;
        }
        
        .badge-credito {
            background: rgba(33, 150, 243, 0.2);
            color: #2196F3;
            border: 1px solid #2196F3;
        }
        
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: var(--color-text-secondary);
        }
        
        .empty-state h3 {
            margin-top: 10px;
            color: var(--color-text);
        }
        
        @media (max-width: 768px) {
            .saque-header {
                flex-direction: column;
                align-items: flex-start;
                gap: 10px;
            }
            
            .saque-info {
                grid-template-columns: 1fr;
            }
            
            .saque-actions {
                flex-direction: column;
            }
            
            .tabs {
                overflow-x: auto;
            }
        }
    </style>
</head>
<body>
    <nav class="navbar">
        <div class="navbar-brand"><i class="fas fa-money-bill-wave"></i> Gerenciar Saques</div>
        <div class="navbar-menu">
            <a href="dashboard.php" class="btn btn-primary">← Voltar</a>
            <a href="logout.php" class="btn btn-danger">Sair</a>
        </div>
    </nav>
    
    <div class="container">
        <?php if ($sucesso): ?>
            <div class="alert alert-success"><?= $sucesso ?></div>
        <?php endif; ?>
        
        <?php if ($erro): ?>
            <div class="alert alert-error"><?= $erro ?></div>
        <?php endif; ?>
        
        <div class="page-header">
            <h1 class="page-title">Gerenciar Saques</h1>
        </div>
        
        <div class="tabs">
            <button class="tab active" onclick="switchTab('pendentes')">
                <i class="fas fa-clock"></i> Pendentes (<?= count($saques_pendentes) ?>)
            </button>
            <button class="tab" onclick="switchTab('aprovados')">
                <i class="fas fa-money-bill-wave"></i> A Pagar (<?= count($saques_aprovados) ?>)
            </button>
            <button class="tab" onclick="switchTab('pagos')">
                <i class="fas fa-check-circle"></i> Histórico (<?= count($saques_pagos) ?>)
            </button>
        </div>
        
        <!-- PENDENTES -->
        <div id="tab-pendentes" class="tab-content active">
            <?php if (count($saques_pendentes) > 0): ?>
                <?php foreach ($saques_pendentes as $saque): ?>
                    <div class="saque-card">
                        <div class="saque-header">
                            <div>
                                <div class="promoter-nome"><?= htmlspecialchars($saque['nome']) ?></div>
                                <div style="margin-top: 5px;">
                                    <?php if ($saque['tipo'] === 'pix'): ?>
                                        <span class="badge badge-pix"><i class="fas fa-pix"></i> PIX</span>
                                    <?php elseif ($saque['tipo'] === 'cartao_consumo'): ?>
                                        <span class="badge badge-credito"><i class="fas fa-credit-card"></i> Cartão de Consumo</span>
                                    <?php else: ?>
                                        <span class="badge badge-credito"><i class="fas fa-glass-cheers"></i> Crédito Consumação</span>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <div class="saque-valor"><?= formatarMoeda($saque['valor']) ?></div>
                        </div>

                        <div class="saque-info">
                            <div class="info-item">
                                <div class="info-label">Solicitado em:</div>
                                <div class="info-value"><?= formatarDataHora($saque['created_at']) ?></div>
                            </div>
                            
                            <div class="info-item">
                                <div class="info-label">Telefone:</div>
                                <div class="info-value">
                                    <a href="https://wa.me/<?= preg_replace('/[^0-9]/', '', $saque['telefone']) ?>" target="_blank" style="color: var(--color-primary);">
                                        <?= htmlspecialchars($saque['telefone']) ?>
                                    </a>
                                </div>
                            </div>
                            
                            <div class="info-item">
                                <div class="info-label">Email:</div>
                                <div class="info-value"><?= htmlspecialchars($saque['email']) ?></div>
                            </div>
                            
                            <?php if ($saque['tipo'] === 'pix' && !empty($saque['pix'])): ?>
                                <div class="info-item">
                                    <div class="info-label">Chave PIX:</div>
                                    <div class="info-value"><?= htmlspecialchars($saque['pix']) ?></div>
                                </div>

                                <?php if (!empty($saque['cpf'])): ?>
                                <div class="info-item">
                                    <div class="info-label">CPF:</div>
                                    <div class="info-value"><?= htmlspecialchars($saque['cpf']) ?></div>
                                </div>
                                <?php endif; ?>
                            <?php endif; ?>
                            
                            <?php if (!empty($saque['eventos'])): ?>
                            <div class="info-item">
                                <div class="info-label">Eventos:</div>
                                <div class="info-value"><?= htmlspecialchars($saque['eventos']) ?></div>
                            </div>
                            <?php endif; ?>
                        </div>
                        
                        <div class="saque-actions">
                            <a href="?aprovar=<?= $saque['id'] ?>" class="btn btn-success" onclick="return confirm('Aprovar este saque?')">
                                <i class="fas fa-check"></i> Aprovar
                            </a>
                            <a href="?rejeitar=<?= $saque['id'] ?>" class="btn btn-danger" onclick="return confirm('Rejeitar este saque? O saldo será devolvido ao promoter.')">
                                <i class="fas fa-times"></i> Rejeitar
                            </a>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php else: ?>
                <div class="empty-state">
                    <div style="font-size: 64px; margin-bottom: 20px; color: #00C851;"><i class="fas fa-check-circle"></i></div>
                    <h3>Nenhum saque pendente</h3>
                </div>
            <?php endif; ?>
        </div>
        
        <!-- A PAGAR -->
        <div id="tab-aprovados" class="tab-content">
            <?php if (count($saques_aprovados) > 0): ?>
                <?php foreach ($saques_aprovados as $saque): ?>
                    <div class="saque-card">
                        <div class="saque-header">
                            <div>
                                <div class="promoter-nome"><?= htmlspecialchars($saque['nome']) ?></div>
                                <div style="margin-top: 5px;">
                                    <?php if ($saque['tipo'] === 'pix'): ?>
                                        <span class="badge badge-pix"><i class="fas fa-pix"></i> PIX</span>
                                    <?php elseif ($saque['tipo'] === 'cartao_consumo'): ?>
                                        <span class="badge badge-credito"><i class="fas fa-credit-card"></i> Cartão de Consumo</span>
                                    <?php else: ?>
                                        <span class="badge badge-credito"><i class="fas fa-glass-cheers"></i> Crédito Consumação</span>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <div class="saque-valor"><?= formatarMoeda($saque['valor']) ?></div>
                        </div>

                        <div class="saque-info">
                            <?php if ($saque['tipo'] === 'pix' && !empty($saque['pix'])): ?>
                                <div class="info-item">
                                    <div class="info-label">Chave PIX:</div>
                                    <div class="info-value"><?= htmlspecialchars($saque['pix']) ?></div>
                                </div>
                                <?php if (!empty($saque['cpf'])): ?>
                                <div class="info-item">
                                    <div class="info-label">CPF:</div>
                                    <div class="info-value"><?= htmlspecialchars($saque['cpf']) ?></div>
                                </div>
                                <?php endif; ?>
                            <?php endif; ?>
                            
                            <div class="info-item">
                                <div class="info-label">Telefone:</div>
                                <div class="info-value"><?= htmlspecialchars($saque['telefone']) ?></div>
                            </div>
                        </div>
                        
                        <div class="saque-actions">
                            <a href="?pagar=<?= $saque['id'] ?>" class="btn btn-success" onclick="return confirm('Confirmar pagamento?')">
                                <i class="fas fa-money-bill-wave"></i> Marcar como PAGO
                            </a>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php else: ?>
                <div class="empty-state">
                    <div style="font-size: 64px; margin-bottom: 20px; color: #FFD700;"><i class="fas fa-money-bill-wave"></i></div>
                    <h3>Nenhum pagamento pendente</h3>
                </div>
            <?php endif; ?>
        </div>
        
        <!-- HISTÓRICO -->
        <div id="tab-pagos" class="tab-content">
            <?php if (count($saques_pagos) > 0): ?>
                <?php foreach ($saques_pagos as $saque): ?>
                    <div class="saque-card">
                        <div class="saque-header">
                            <div>
                                <div class="promoter-nome"><?= htmlspecialchars($saque['nome']) ?></div>
                                <div style="margin-top: 5px;">
                                    <span class="badge" style="background: #00C851;"><i class="fas fa-check"></i> PAGO</span>
                                    <?php if ($saque['tipo'] === 'pix'): ?>
                                        <span class="badge badge-pix"><i class="fas fa-pix"></i> PIX</span>
                                    <?php elseif ($saque['tipo'] === 'cartao_consumo'): ?>
                                        <span class="badge badge-credito"><i class="fas fa-credit-card"></i> Cartão de Consumo</span>
                                    <?php else: ?>
                                        <span class="badge badge-credito"><i class="fas fa-glass-cheers"></i> Crédito</span>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <div class="saque-valor"><?= formatarMoeda($saque['valor']) ?></div>
                        </div>
                        
                        <div class="saque-info">
                            <div class="info-item">
                                <div class="info-label">Pago em:</div>
                                <div class="info-value"><?= formatarDataHora($saque['processado_at']) ?></div>
                            </div>
                            
                            <?php if (!empty($saque['eventos'])): ?>
                            <div class="info-item">
                                <div class="info-label">Eventos:</div>
                                <div class="info-value"><?= htmlspecialchars($saque['eventos']) ?></div>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php else: ?>
                <div class="empty-state">
                    <div style="font-size: 64px; margin-bottom: 20px; color: #FFD700;"><i class="fas fa-clipboard-list"></i></div>
                    <h3>Nenhum pagamento encontrado</h3>
                </div>
            <?php endif; ?>
        </div>
    </div>
    
    <script>
        const savedTheme = localStorage.getItem('theme') || 'dark';
        document.documentElement.setAttribute('data-theme', savedTheme);
        
        function switchTab(tabName) {
            document.querySelectorAll('.tab').forEach(t => t.classList.remove('active'));
            document.querySelectorAll('.tab-content').forEach(c => c.classList.remove('active'));
            
            event.target.classList.add('active');
            document.getElementById('tab-' + tabName).classList.add('active');
        }
    </script>
</body>
</html>