<?php
/**
 * CONFIGURAÇÕES DO SISTEMA - WHITE LABEL v2.0
 * Sistema completo de personalização visual e funcional
 */
require_once 'config.php';

// Verificar se é admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: dashboard.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$sucesso = '';
$erro = '';

// Buscar dados do usuário para navbar
$stmt = $pdo->prepare("SELECT * FROM usuarios WHERE id = ?");
$stmt->execute([$user_id]);
$usuario = $stmt->fetch();

// Buscar configurações visuais atuais
try {
    $stmt = $pdo->query("SELECT * FROM configuracoes_visual WHERE id = 1");
    $config_visual = $stmt->fetch();

    if (!$config_visual) {
        // Criar configuração padrão se não existir
        $pdo->exec("INSERT INTO configuracoes_visual (id, nome_sistema, paleta_ativa) VALUES (1, 'PromoTaal', 'atual')");
        $stmt = $pdo->query("SELECT * FROM configuracoes_visual WHERE id = 1");
        $config_visual = $stmt->fetch();
    }
} catch (PDOException $e) {
    $erro = "⚠️ Execute a migration primeiro: migration_whitelabel_v2.sql ou use o instalador: install.php";
    $config_visual = false;
}

// Buscar todas as paletas disponíveis
$paletas = [];
if ($config_visual) {
    try {
        $stmt = $pdo->query("SELECT * FROM paletas_cores ORDER BY tipo, nome");
        $paletas = $stmt->fetchAll();
    } catch (PDOException $e) {
        // Tabela não existe ainda
    }
}

// Processar formulário
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $nome_sistema = filter_input(INPUT_POST, 'nome_sistema', FILTER_SANITIZE_STRING);
        $tipo_menu = filter_input(INPUT_POST, 'tipo_menu', FILTER_SANITIZE_STRING);
        $paleta_ativa = filter_input(INPUT_POST, 'paleta_ativa', FILTER_SANITIZE_STRING);

        // Cores individuais (caso seja personalização manual)
        $cor_primaria = filter_input(INPUT_POST, 'cor_primaria', FILTER_SANITIZE_STRING);
        $cor_secundaria = filter_input(INPUT_POST, 'cor_secundaria', FILTER_SANITIZE_STRING);
        $cor_fundo = filter_input(INPUT_POST, 'cor_fundo', FILTER_SANITIZE_STRING);
        $cor_card = filter_input(INPUT_POST, 'cor_card', FILTER_SANITIZE_STRING);
        $cor_texto = filter_input(INPUT_POST, 'cor_texto', FILTER_SANITIZE_STRING);
        $cor_texto_secundario = filter_input(INPUT_POST, 'cor_texto_secundario', FILTER_SANITIZE_STRING);
        $cor_borda = filter_input(INPUT_POST, 'cor_borda', FILTER_SANITIZE_STRING);
        $cor_sucesso = filter_input(INPUT_POST, 'cor_sucesso', FILTER_SANITIZE_STRING);
        $cor_erro = filter_input(INPUT_POST, 'cor_erro', FILTER_SANITIZE_STRING);
        $cor_aviso = filter_input(INPUT_POST, 'cor_aviso', FILTER_SANITIZE_STRING);
        $cor_info = filter_input(INPUT_POST, 'cor_info', FILTER_SANITIZE_STRING);
        $cor_hover = filter_input(INPUT_POST, 'cor_hover', FILTER_SANITIZE_STRING);

        $logo_path = $config_visual['logo_path'];

        // Upload do logo
        if (isset($_FILES['logo_sistema']) && $_FILES['logo_sistema']['error'] == 0) {
            $allowed = ['jpg', 'jpeg', 'png', 'gif', 'svg'];
            $filename = $_FILES['logo_sistema']['name'];
            $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));

            if (in_array($ext, $allowed) && $_FILES['logo_sistema']['size'] <= 5242880) { // 5MB
                $novo_nome = 'logo_' . time() . '.' . $ext;
                $destino = 'uploads/sistema/' . $novo_nome;

                if (!is_dir('uploads/sistema')) {
                    mkdir('uploads/sistema', 0777, true);
                }

                if (move_uploaded_file($_FILES['logo_sistema']['tmp_name'], $destino)) {
                    // Deletar logo antigo se existir
                    if (!empty($logo_path) && file_exists($logo_path)) {
                        unlink($logo_path);
                    }
                    $logo_path = $destino;
                }
            } else {
                $erro = 'Logo inválido. Use JPG, PNG, GIF ou SVG (máx 5MB)';
            }
        }

        // Aplicar paleta se selecionada
        if ($paleta_ativa && $paleta_ativa !== 'personalizada') {
            $stmt = $pdo->prepare("SELECT * FROM paletas_cores WHERE nome = ?");
            $stmt->execute([$paleta_ativa]);
            $paleta = $stmt->fetch();

            if ($paleta) {
                $cor_primaria = $paleta['cor_primaria'];
                $cor_secundaria = $paleta['cor_secundaria'];
                $cor_fundo = $paleta['cor_fundo'];
                $cor_card = $paleta['cor_card'];
                $cor_texto = $paleta['cor_texto'];
                $cor_texto_secundario = $paleta['cor_texto_secundario'];
                $cor_borda = $paleta['cor_borda'];
                $cor_sucesso = $paleta['cor_sucesso'];
                $cor_erro = $paleta['cor_erro'];
                $cor_aviso = $paleta['cor_aviso'];
                $cor_info = $paleta['cor_info'];
                $cor_hover = $paleta['cor_hover'];
            }
        }

        // Atualizar no banco
        $stmt = $pdo->prepare("
            UPDATE configuracoes_visual SET
                nome_sistema = ?,
                logo_path = ?,
                cor_primaria = ?,
                cor_secundaria = ?,
                cor_fundo = ?,
                cor_card = ?,
                cor_texto = ?,
                cor_texto_secundario = ?,
                cor_borda = ?,
                cor_sucesso = ?,
                cor_erro = ?,
                cor_aviso = ?,
                cor_info = ?,
                cor_hover = ?,
                tipo_menu = ?,
                paleta_ativa = ?
            WHERE id = 1
        ");

        if ($stmt->execute([
            $nome_sistema, $logo_path,
            $cor_primaria, $cor_secundaria, $cor_fundo, $cor_card,
            $cor_texto, $cor_texto_secundario, $cor_borda,
            $cor_sucesso, $cor_erro, $cor_aviso, $cor_info, $cor_hover,
            $tipo_menu, $paleta_ativa
        ])) {
            $sucesso = '✓ Configurações salvas com sucesso! As alterações serão aplicadas em todo o sistema.';
            // Recarregar configurações
            $stmt = $pdo->query("SELECT * FROM configuracoes_visual WHERE id = 1");
            $config_visual = $stmt->fetch();
        } else {
            $erro = 'Erro ao salvar configurações.';
        }
    } catch (PDOException $e) {
        $erro = 'Erro: ' . $e->getMessage();
    }
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Configurações - White Label v2.0</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="assets/dashboard-style.css">
    <style><?php include 'sistema_theme.php'; ?></style>
    <style>
        body {
            background: var(--cor-fundo);
            min-height: 100vh;
            padding: 20px;
        }

        .config-container {
            max-width: 1400px;
            margin: 0 auto;
        }

        .header {
            text-align: center;
            margin-bottom: 40px;
            padding: 30px;
            background: linear-gradient(135deg, var(--cor-card), var(--cor-secundaria));
            border-radius: 20px;
            border: 1px solid var(--cor-borda);
        }

        .header-title {
            font-size: 42px;
            font-weight: 700;
            color: var(--cor-primaria);
            margin-bottom: 10px;
        }

        .header-subtitle {
            color: var(--cor-texto-secundario);
            font-size: 18px;
        }

        .version-badge {
            display: inline-block;
            background: var(--cor-primaria);
            color: #000;
            padding: 5px 15px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 700;
            margin-top: 10px;
        }

        .config-sections {
            display: grid;
            gap: 30px;
        }

        .config-section {
            background: var(--cor-card);
            border-radius: 20px;
            padding: 30px;
            border: 1px solid var(--cor-borda);
        }

        .section-header {
            display: flex;
            align-items: center;
            gap: 15px;
            margin-bottom: 25px;
            padding-bottom: 20px;
            border-bottom: 2px solid var(--cor-borda);
        }

        .section-icon {
            width: 50px;
            height: 50px;
            background: var(--cor-primaria);
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            color: #000;
        }

        .section-title {
            font-size: 26px;
            font-weight: 700;
            color: var(--cor-texto);
        }

        .section-description {
            color: var(--cor-texto-secundario);
            font-size: 14px;
            margin-top: 5px;
        }

        .paletas-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }

        .paleta-card {
            background: var(--cor-secundaria);
            border: 3px solid var(--cor-borda);
            border-radius: 15px;
            padding: 20px;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .paleta-card:hover {
            transform: translateY(-5px);
            border-color: var(--cor-primaria);
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
        }

        .paleta-card.active {
            border-color: var(--cor-primaria);
            background: linear-gradient(135deg, var(--cor-secundaria), var(--cor-card));
            box-shadow: 0 0 20px var(--cor-primaria);
        }

        .paleta-nome {
            font-size: 18px;
            font-weight: 700;
            color: var(--cor-texto);
            margin-bottom: 8px;
        }

        .paleta-descricao {
            font-size: 13px;
            color: var(--cor-texto-secundario);
            margin-bottom: 15px;
        }

        .paleta-tipo-badge {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
            margin-bottom: 15px;
        }

        .paleta-tipo-badge.escura {
            background: #1a1a1a;
            color: #fff;
        }

        .paleta-tipo-badge.clara {
            background: #f5f5f5;
            color: #000;
        }

        .paleta-cores {
            display: grid;
            grid-template-columns: repeat(6, 1fr);
            gap: 5px;
        }

        .paleta-cor {
            height: 35px;
            border-radius: 8px;
            border: 2px solid var(--cor-borda);
        }

        .cores-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }

        .cor-item {
            background: var(--cor-secundaria);
            padding: 20px;
            border-radius: 12px;
            border: 1px solid var(--cor-borda);
        }

        .cor-label {
            font-weight: 700;
            color: var(--cor-texto);
            margin-bottom: 8px;
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .cor-descricao {
            font-size: 12px;
            color: var(--cor-texto-secundario);
            margin-bottom: 12px;
            line-height: 1.5;
        }

        .cor-picker-wrapper {
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .cor-preview {
            width: 60px;
            height: 60px;
            border-radius: 10px;
            border: 3px solid var(--cor-borda);
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .cor-preview:hover {
            transform: scale(1.1);
            border-color: var(--cor-primaria);
        }

        input[type="color"] {
            opacity: 0;
            width: 0;
            height: 0;
            position: absolute;
        }

        .cor-value {
            flex: 1;
            background: var(--cor-card);
            border: 2px solid var(--cor-borda);
            padding: 12px 15px;
            border-radius: 8px;
            color: var(--cor-texto);
            font-family: monospace;
            font-size: 14px;
            font-weight: 600;
        }

        .menu-options {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }

        .menu-option {
            background: var(--cor-secundaria);
            border: 3px solid var(--cor-borda);
            border-radius: 15px;
            padding: 25px;
            cursor: pointer;
            transition: all 0.3s ease;
            text-align: center;
        }

        .menu-option:hover {
            border-color: var(--cor-primaria);
            transform: translateY(-5px);
        }

        .menu-option.active {
            border-color: var(--cor-primaria);
            background: linear-gradient(135deg, var(--cor-secundaria), var(--cor-card));
            box-shadow: 0 0 20px var(--cor-primaria);
        }

        .menu-option-icon {
            font-size: 48px;
            color: var(--cor-primaria);
            margin-bottom: 15px;
        }

        .menu-option-nome {
            font-size: 20px;
            font-weight: 700;
            color: var(--cor-texto);
            margin-bottom: 10px;
        }

        .menu-option-desc {
            font-size: 13px;
            color: var(--cor-texto-secundario);
        }

        .logo-upload {
            background: var(--cor-secundaria);
            border: 2px dashed var(--cor-borda);
            border-radius: 12px;
            padding: 30px;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .logo-upload:hover {
            border-color: var(--cor-primaria);
            background: color-mix(in srgb, var(--cor-primaria) 5%, var(--cor-secundaria));
        }

        .logo-preview {
            max-width: 300px;
            max-height: 150px;
            margin: 20px auto;
            display: block;
            border-radius: 10px;
        }

        .actions {
            display: flex;
            gap: 20px;
            margin-top: 40px;
        }

        .btn-save {
            flex: 1;
            background: var(--cor-primaria);
            color: #000;
            border: none;
            padding: 18px 40px;
            border-radius: 12px;
            font-size: 18px;
            font-weight: 700;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .btn-save:hover {
            background: var(--cor-hover);
            transform: translateY(-2px);
            box-shadow: 0 10px 30px rgba(255, 193, 7, 0.3);
        }

        .btn-back {
            background: var(--cor-secundaria);
            color: var(--cor-texto);
            border: 2px solid var(--cor-borda);
            padding: 18px 40px;
            border-radius: 12px;
            font-size: 18px;
            font-weight: 700;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 10px;
            transition: all 0.3s ease;
        }

        .btn-back:hover {
            border-color: var(--cor-primaria);
            color: var(--cor-primaria);
        }

        @media (max-width: 768px) {
            .header-title {
                font-size: 28px;
            }

            .paletas-grid,
            .cores-grid,
            .menu-options {
                grid-template-columns: 1fr;
            }

            .actions {
                flex-direction: column;
            }
        }

        .alert {
            padding: 20px 25px;
            border-radius: 12px;
            margin-bottom: 25px;
            display: flex;
            align-items: center;
            gap: 15px;
            font-size: 16px;
            animation: slideDown 0.4s ease;
        }

        .alert-success {
            background: color-mix(in srgb, var(--cor-sucesso) 15%, var(--cor-card));
            border: 2px solid var(--cor-sucesso);
            color: var(--cor-sucesso);
        }

        .alert-error {
            background: color-mix(in srgb, var(--cor-erro) 15%, var(--cor-card));
            border: 2px solid var(--cor-erro);
            color: var(--cor-erro);
        }

        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateY(-20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .info-box {
            background: color-mix(in srgb, var(--cor-info) 10%, var(--cor-card));
            border-left: 4px solid var(--cor-info);
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            color: var(--cor-texto-secundario);
            font-size: 14px;
            line-height: 1.6;
        }

        .info-box strong {
            color: var(--cor-info);
        }
    </style>
</head>
<body>
    <?php include 'assets/navbar.php'; ?>

    <div class="config-container">
        <div class="header">
            <h1 class="header-title">
                <i class="fas fa-palette"></i> Configurações White Label
            </h1>
            <p class="header-subtitle">Personalize completamente a aparência e funcionalidades do sistema</p>
            <span class="version-badge">VERSÃO 2.0</span>
        </div>

        <?php if ($sucesso): ?>
            <div class="alert alert-success">
                <i class="fas fa-check-circle" style="font-size: 24px;"></i>
                <span><?= $sucesso ?></span>
            </div>
        <?php endif; ?>

        <?php if ($erro): ?>
            <div class="alert alert-error">
                <i class="fas fa-exclamation-triangle" style="font-size: 24px;"></i>
                <span><?= $erro ?></span>
            </div>
        <?php endif; ?>

        <form method="POST" enctype="multipart/form-data" id="configForm">
            <div class="config-sections">

                <!-- 1. INFORMAÇÕES BÁSICAS -->
                <div class="config-section">
                    <div class="section-header">
                        <div class="section-icon"><i class="fas fa-building"></i></div>
                        <div>
                            <h2 class="section-title">Informações Básicas</h2>
                            <p class="section-description">Nome do sistema e identidade visual</p>
                        </div>
                    </div>

                    <div class="form-group">
                        <label class="cor-label">
                            <i class="fas fa-signature"></i>
                            Nome do Sistema
                        </label>
                        <input type="text" name="nome_sistema" class="cor-value" style="width: 100%;"
                               value="<?= htmlspecialchars($config_visual['nome_sistema']) ?>" required
                               placeholder="Ex: PromoTaal, MeuEvento, etc.">
                    </div>

                    <div class="form-group" style="margin-top: 25px;">
                        <label class="cor-label">
                            <i class="fas fa-image"></i>
                            Logo do Sistema
                        </label>
                        <p class="cor-descricao">Formatos aceitos: JPG, PNG, GIF, SVG (máximo 5MB)</p>
                        <label for="logo_sistema" class="logo-upload">
                            <i class="fas fa-cloud-upload-alt" style="font-size: 48px; color: var(--cor-primaria); margin-bottom: 15px;"></i>
                            <p style="color: var(--cor-texto); font-weight: 600; margin-bottom: 5px;">Clique para fazer upload do logo</p>
                            <p style="color: var(--cor-texto-secundario); font-size: 13px;">ou arraste e solte aqui</p>
                        </label>
                        <input type="file" id="logo_sistema" name="logo_sistema" accept="image/*" style="display: none;" onchange="previewLogo(event)">
                        <?php if (!empty($config_visual['logo_path'])): ?>
                            <img src="<?= htmlspecialchars($config_visual['logo_path']) ?>" class="logo-preview" id="logo-preview">
                        <?php endif; ?>
                    </div>
                </div>

                <!-- 2. PALETAS DE CORES PRÉ-DEFINIDAS -->
                <div class="config-section">
                    <div class="section-header">
                        <div class="section-icon"><i class="fas fa-swatchbook"></i></div>
                        <div>
                            <h2 class="section-title">Paletas de Cores</h2>
                            <p class="section-description">Escolha uma paleta pré-definida ou personalize manualmente</p>
                        </div>
                    </div>

                    <div class="info-box">
                        <strong><i class="fas fa-info-circle"></i> Dica:</strong> Selecione uma paleta abaixo para aplicar instantaneamente. Para personalizar cores individuais, escolha "Personalizada" e edite a seção seguinte.
                    </div>

                    <div class="paletas-grid">
                        <?php foreach ($paletas as $paleta): ?>
                            <label class="paleta-card <?= $config_visual['paleta_ativa'] === $paleta['nome'] ? 'active' : '' ?>">
                                <input type="radio" name="paleta_ativa" value="<?= $paleta['nome'] ?>"
                                       <?= $config_visual['paleta_ativa'] === $paleta['nome'] ? 'checked' : '' ?>
                                       style="display: none;" onchange="aplicarPaleta(this)">

                                <div class="paleta-nome"><?= htmlspecialchars($paleta['nome']) ?></div>
                                <span class="paleta-tipo-badge <?= $paleta['tipo'] ?>"><?= ucfirst($paleta['tipo']) ?></span>
                                <div class="paleta-descricao"><?= htmlspecialchars($paleta['descricao']) ?></div>

                                <div class="paleta-cores">
                                    <div class="paleta-cor" style="background-color: <?= $paleta['cor_primaria'] ?>"></div>
                                    <div class="paleta-cor" style="background-color: <?= $paleta['cor_fundo'] ?>"></div>
                                    <div class="paleta-cor" style="background-color: <?= $paleta['cor_card'] ?>"></div>
                                    <div class="paleta-cor" style="background-color: <?= $paleta['cor_sucesso'] ?>"></div>
                                    <div class="paleta-cor" style="background-color: <?= $paleta['cor_erro'] ?>"></div>
                                    <div class="paleta-cor" style="background-color: <?= $paleta['cor_info'] ?>"></div>
                                </div>
                            </label>
                        <?php endforeach; ?>

                        <!-- Opção Personalizada -->
                        <label class="paleta-card <?= $config_visual['paleta_ativa'] === 'personalizada' ? 'active' : '' ?>">
                            <input type="radio" name="paleta_ativa" value="personalizada"
                                   <?= $config_visual['paleta_ativa'] === 'personalizada' ? 'checked' : '' ?>
                                   style="display: none;" onchange="aplicarPaleta(this)">
                            <div class="paleta-nome">✨ Personalizada</div>
                            <span class="paleta-tipo-badge" style="background: linear-gradient(90deg, #667eea 0%, #764ba2 100%); color: #fff;">Custom</span>
                            <div class="paleta-descricao">Crie sua própria combinação de cores editando manualmente cada elemento</div>
                            <div style="text-align: center; padding: 20px;">
                                <i class="fas fa-paint-brush" style="font-size: 40px; color: var(--cor-primaria);"></i>
                            </div>
                        </label>
                    </div>
                </div>

                <!-- 3. PERSONALIZAÇÃO INDIVIDUAL DE CORES -->
                <div class="config-section">
                    <div class="section-header">
                        <div class="section-icon"><i class="fas fa-sliders-h"></i></div>
                        <div>
                            <h2 class="section-title">Cores Personalizadas</h2>
                            <p class="section-description">Ajuste cada cor individualmente para criar sua identidade visual</p>
                        </div>
                    </div>

                    <div class="info-box">
                        <strong><i class="fas fa-lightbulb"></i> Como funciona:</strong> Cada cor abaixo controla elementos específicos do sistema. Clique no quadrado colorido para escolher uma nova cor.
                    </div>

                    <div class="cores-grid">
                        <!-- Cor Primária -->
                        <div class="cor-item">
                            <div class="cor-label">
                                <i class="fas fa-star"></i>
                                Cor Primária
                            </div>
                            <div class="cor-descricao">
                                <strong>Usada em:</strong> Botões principais, links importantes, destaques, badges, hover effects. É a cor de identidade do sistema.
                            </div>
                            <div class="cor-picker-wrapper">
                                <label for="cor_primaria" class="cor-preview" style="background-color: <?= $config_visual['cor_primaria'] ?>"></label>
                                <input type="color" id="cor_primaria" name="cor_primaria" value="<?= $config_visual['cor_primaria'] ?>" onchange="updateColorPreview(this)">
                                <input type="text" class="cor-value" value="<?= $config_visual['cor_primaria'] ?>" readonly>
                            </div>
                        </div>

                        <!-- Cor Secundária -->
                        <div class="cor-item">
                            <div class="cor-label">
                                <i class="fas fa-square"></i>
                                Cor Secundária
                            </div>
                            <div class="cor-descricao">
                                <strong>Usada em:</strong> Fundos de elementos secundários, headers, menus, rodapés. Complementa a cor primária.
                            </div>
                            <div class="cor-picker-wrapper">
                                <label for="cor_secundaria" class="cor-preview" style="background-color: <?= $config_visual['cor_secundaria'] ?>"></label>
                                <input type="color" id="cor_secundaria" name="cor_secundaria" value="<?= $config_visual['cor_secundaria'] ?>" onchange="updateColorPreview(this)">
                                <input type="text" class="cor-value" value="<?= $config_visual['cor_secundaria'] ?>" readonly>
                            </div>
                        </div>

                        <!-- Cor de Fundo -->
                        <div class="cor-item">
                            <div class="cor-label">
                                <i class="fas fa-fill-drip"></i>
                                Cor de Fundo
                            </div>
                            <div class="cor-descricao">
                                <strong>Usada em:</strong> Fundo principal de todas as páginas (body). Base visual do sistema.
                            </div>
                            <div class="cor-picker-wrapper">
                                <label for="cor_fundo" class="cor-preview" style="background-color: <?= $config_visual['cor_fundo'] ?>"></label>
                                <input type="color" id="cor_fundo" name="cor_fundo" value="<?= $config_visual['cor_fundo'] ?>" onchange="updateColorPreview(this)">
                                <input type="text" class="cor-value" value="<?= $config_visual['cor_fundo'] ?>" readonly>
                            </div>
                        </div>

                        <!-- Cor dos Cards -->
                        <div class="cor-item">
                            <div class="cor-label">
                                <i class="fas fa-clone"></i>
                                Cor dos Cards
                            </div>
                            <div class="cor-descricao">
                                <strong>Usada em:</strong> Cards, modais, containers, caixas de conteúdo, formulários, tabelas.
                            </div>
                            <div class="cor-picker-wrapper">
                                <label for="cor_card" class="cor-preview" style="background-color: <?= $config_visual['cor_card'] ?>"></label>
                                <input type="color" id="cor_card" name="cor_card" value="<?= $config_visual['cor_card'] ?>" onchange="updateColorPreview(this)">
                                <input type="text" class="cor-value" value="<?= $config_visual['cor_card'] ?>" readonly>
                            </div>
                        </div>

                        <!-- Cor do Texto -->
                        <div class="cor-item">
                            <div class="cor-label">
                                <i class="fas fa-font"></i>
                                Cor do Texto Principal
                            </div>
                            <div class="cor-descricao">
                                <strong>Usada em:</strong> Textos principais, títulos (H1-H6), parágrafos importantes, labels principais.
                            </div>
                            <div class="cor-picker-wrapper">
                                <label for="cor_texto" class="cor-preview" style="background-color: <?= $config_visual['cor_texto'] ?>"></label>
                                <input type="color" id="cor_texto" name="cor_texto" value="<?= $config_visual['cor_texto'] ?>" onchange="updateColorPreview(this)">
                                <input type="text" class="cor-value" value="<?= $config_visual['cor_texto'] ?>" readonly>
                            </div>
                        </div>

                        <!-- Cor do Texto Secundário -->
                        <div class="cor-item">
                            <div class="cor-label">
                                <i class="fas fa-text-height"></i>
                                Cor do Texto Secundário
                            </div>
                            <div class="cor-descricao">
                                <strong>Usada em:</strong> Textos de apoio, descrições, placeholders, labels secundários, datas.
                            </div>
                            <div class="cor-picker-wrapper">
                                <label for="cor_texto_secundario" class="cor-preview" style="background-color: <?= $config_visual['cor_texto_secundario'] ?>"></label>
                                <input type="color" id="cor_texto_secundario" name="cor_texto_secundario" value="<?= $config_visual['cor_texto_secundario'] ?>" onchange="updateColorPreview(this)">
                                <input type="text" class="cor-value" value="<?= $config_visual['cor_texto_secundario'] ?>" readonly>
                            </div>
                        </div>

                        <!-- Cor das Bordas -->
                        <div class="cor-item">
                            <div class="cor-label">
                                <i class="fas fa-border-style"></i>
                                Cor das Bordas
                            </div>
                            <div class="cor-descricao">
                                <strong>Usada em:</strong> Bordas de inputs, cards, tabelas, separadores (hr), divisores.
                            </div>
                            <div class="cor-picker-wrapper">
                                <label for="cor_borda" class="cor-preview" style="background-color: <?= $config_visual['cor_borda'] ?>"></label>
                                <input type="color" id="cor_borda" name="cor_borda" value="<?= $config_visual['cor_borda'] ?>" onchange="updateColorPreview(this)">
                                <input type="text" class="cor-value" value="<?= $config_visual['cor_borda'] ?>" readonly>
                            </div>
                        </div>

                        <!-- Cor de Sucesso -->
                        <div class="cor-item">
                            <div class="cor-label">
                                <i class="fas fa-check-circle"></i>
                                Cor de Sucesso
                            </div>
                            <div class="cor-descricao">
                                <strong>Usada em:</strong> Mensagens de sucesso, status "concluído", "aprovado", botões de confirmação.
                            </div>
                            <div class="cor-picker-wrapper">
                                <label for="cor_sucesso" class="cor-preview" style="background-color: <?= $config_visual['cor_sucesso'] ?>"></label>
                                <input type="color" id="cor_sucesso" name="cor_sucesso" value="<?= $config_visual['cor_sucesso'] ?>" onchange="updateColorPreview(this)">
                                <input type="text" class="cor-value" value="<?= $config_visual['cor_sucesso'] ?>" readonly>
                            </div>
                        </div>

                        <!-- Cor de Erro -->
                        <div class="cor-item">
                            <div class="cor-label">
                                <i class="fas fa-exclamation-circle"></i>
                                Cor de Erro
                            </div>
                            <div class="cor-descricao">
                                <strong>Usada em:</strong> Mensagens de erro, status "cancelado", "rejeitado", alertas críticos, botões de exclusão.
                            </div>
                            <div class="cor-picker-wrapper">
                                <label for="cor_erro" class="cor-preview" style="background-color: <?= $config_visual['cor_erro'] ?>"></label>
                                <input type="color" id="cor_erro" name="cor_erro" value="<?= $config_visual['cor_erro'] ?>" onchange="updateColorPreview(this)">
                                <input type="text" class="cor-value" value="<?= $config_visual['cor_erro'] ?>" readonly>
                            </div>
                        </div>

                        <!-- Cor de Aviso -->
                        <div class="cor-item">
                            <div class="cor-label">
                                <i class="fas fa-exclamation-triangle"></i>
                                Cor de Aviso
                            </div>
                            <div class="cor-descricao">
                                <strong>Usada em:</strong> Avisos importantes, status "pendente", alertas de atenção, notificações.
                            </div>
                            <div class="cor-picker-wrapper">
                                <label for="cor_aviso" class="cor-preview" style="background-color: <?= $config_visual['cor_aviso'] ?>"></label>
                                <input type="color" id="cor_aviso" name="cor_aviso" value="<?= $config_visual['cor_aviso'] ?>" onchange="updateColorPreview(this)">
                                <input type="text" class="cor-value" value="<?= $config_visual['cor_aviso'] ?>" readonly>
                            </div>
                        </div>

                        <!-- Cor de Info -->
                        <div class="cor-item">
                            <div class="cor-label">
                                <i class="fas fa-info-circle"></i>
                                Cor de Informação
                            </div>
                            <div class="cor-descricao">
                                <strong>Usada em:</strong> Mensagens informativas, status "processando", dicas, tooltips, ajuda.
                            </div>
                            <div class="cor-picker-wrapper">
                                <label for="cor_info" class="cor-preview" style="background-color: <?= $config_visual['cor_info'] ?>"></label>
                                <input type="color" id="cor_info" name="cor_info" value="<?= $config_visual['cor_info'] ?>" onchange="updateColorPreview(this)">
                                <input type="text" class="cor-value" value="<?= $config_visual['cor_info'] ?>" readonly>
                            </div>
                        </div>

                        <!-- Cor de Hover -->
                        <div class="cor-item">
                            <div class="cor-label">
                                <i class="fas fa-hand-pointer"></i>
                                Cor de Hover
                            </div>
                            <div class="cor-descricao">
                                <strong>Usada em:</strong> Efeito ao passar o mouse sobre botões, links, cards clicáveis. Realce interativo.
                            </div>
                            <div class="cor-picker-wrapper">
                                <label for="cor_hover" class="cor-preview" style="background-color: <?= $config_visual['cor_hover'] ?>"></label>
                                <input type="color" id="cor_hover" name="cor_hover" value="<?= $config_visual['cor_hover'] ?>" onchange="updateColorPreview(this)">
                                <input type="text" class="cor-value" value="<?= $config_visual['cor_hover'] ?>" readonly>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- 4. TIPO DE MENU -->
                <div class="config-section">
                    <div class="section-header">
                        <div class="section-icon"><i class="fas fa-bars"></i></div>
                        <div>
                            <h2 class="section-title">Tipo de Menu</h2>
                            <p class="section-description">Escolha o layout de navegação que melhor se adapta ao seu negócio</p>
                        </div>
                    </div>

                    <div class="menu-options">
                        <!-- Menu Tipo 1 -->
                        <label class="menu-option <?= $config_visual['tipo_menu'] === 'menu_lateral' ? 'active' : '' ?>">
                            <input type="radio" name="tipo_menu" value="menu_lateral"
                                   <?= $config_visual['tipo_menu'] === 'menu_lateral' ? 'checked' : '' ?>
                                   style="display: none;">
                            <div class="menu-option-icon"><i class="fas fa-bars"></i></div>
                            <div class="menu-option-nome">Menu Lateral Clássico</div>
                            <div class="menu-option-desc">Navbar superior com menu lateral em desktop. Ideal para muitas opções.</div>
                        </label>

                        <!-- Menu Tipo 2 -->
                        <label class="menu-option <?= $config_visual['tipo_menu'] === 'menu_rodape' ? 'active' : '' ?>">
                            <input type="radio" name="tipo_menu" value="menu_rodape"
                                   <?= $config_visual['tipo_menu'] === 'menu_rodape' ? 'checked' : '' ?>
                                   style="display: none;">
                            <div class="menu-option-icon"><i class="fas fa-grip-horizontal"></i></div>
                            <div class="menu-option-nome">Menu Fixo no Rodapé</div>
                            <div class="menu-option-desc">Atalhos rápidos fixos no rodapé. Personalizável por usuário. Ótimo para mobile.</div>
                        </label>

                        <!-- Menu Tipo 3 -->
                        <label class="menu-option <?= $config_visual['tipo_menu'] === 'menu_banco' ? 'active' : '' ?>">
                            <input type="radio" name="tipo_menu" value="menu_banco"
                                   <?= $config_visual['tipo_menu'] === 'menu_banco' ? 'checked' : '' ?>
                                   style="display: none;">
                            <div class="menu-option-icon"><i class="fas fa-th"></i></div>
                            <div class="menu-option-nome">Menu Estilo Banco</div>
                            <div class="menu-option-desc">Grid moderno de ícones grandes. Navegação rápida mobile-first inspirada em apps bancários.</div>
                        </label>
                    </div>
                </div>

            </div>

            <!-- AÇÕES -->
            <div class="actions">
                <button type="submit" class="btn-save">
                    <i class="fas fa-save"></i> Salvar Todas as Configurações
                </button>
                <a href="dashboard.php" class="btn-back">
                    <i class="fas fa-arrow-left"></i> Voltar ao Dashboard
                </a>
            </div>
        </form>
    </div>

    <script>
        // Preview do logo
        function previewLogo(event) {
            const file = event.target.files[0];
            if (file) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    let preview = document.getElementById('logo-preview');
                    if (!preview) {
                        preview = document.createElement('img');
                        preview.id = 'logo-preview';
                        preview.className = 'logo-preview';
                        event.target.parentElement.appendChild(preview);
                    }
                    preview.src = e.target.result;
                }
                reader.readAsDataURL(file);
            }
        }

        // Atualizar preview de cor
        function updateColorPreview(input) {
            const preview = input.previousElementSibling;
            const textInput = input.nextElementSibling;
            preview.style.backgroundColor = input.value;
            textInput.value = input.value;
        }

        // Aplicar paleta selecionada
        function aplicarPaleta(radio) {
            // Adicionar classe active no card
            document.querySelectorAll('.paleta-card').forEach(card => card.classList.remove('active'));
            radio.closest('.paleta-card').classList.add('active');
        }

        // Adicionar classe active nos cards de menu
        document.querySelectorAll('.menu-option input').forEach(input => {
            input.addEventListener('change', function() {
                document.querySelectorAll('.menu-option').forEach(opt => opt.classList.remove('active'));
                this.closest('.menu-option').classList.add('active');
            });
        });

        // Click no preview abre o color picker
        document.querySelectorAll('.cor-preview').forEach(preview => {
            preview.addEventListener('click', function() {
                this.nextElementSibling.click();
            });
        });
    </script>
</body>
</html>
