<?php
// debug_cadastro.php - Diagnosticar erro no cadastro
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once 'config.php';

?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <title>Debug - Cadastro de Usuários</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: Arial, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 20px;
            min-height: 100vh;
        }
        .container {
            max-width: 1000px;
            margin: 0 auto;
            background: white;
            border-radius: 20px;
            padding: 40px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
        }
        h1 { color: #667eea; margin-bottom: 20px; }
        h2 { color: #764ba2; margin: 25px 0 15px 0; border-bottom: 2px solid #667eea; padding-bottom: 10px; }
        .success { background: #d4edda; color: #155724; padding: 15px; margin: 15px 0; border-radius: 5px; border-left: 4px solid #28a745; }
        .error { background: #f8d7da; color: #721c24; padding: 15px; margin: 15px 0; border-radius: 5px; border-left: 4px solid #dc3545; }
        .info { background: #d1ecf1; color: #0c5460; padding: 15px; margin: 15px 0; border-radius: 5px; border-left: 4px solid #17a2b8; }
        pre { background: #f8f9fa; padding: 15px; border-radius: 5px; overflow-x: auto; font-size: 12px; border: 1px solid #dee2e6; }
        table { width: 100%; border-collapse: collapse; margin: 15px 0; }
        table th { background: #667eea; color: white; padding: 10px; text-align: left; font-size: 12px; }
        table td { padding: 8px 10px; border-bottom: 1px solid #ddd; font-size: 12px; }
    </style>
</head>
<body>
<div class="container">
    <h1>🔍 Debug - Cadastro de Usuários</h1>

<?php
try {
    echo "<div class='success'>";
    echo "<p>✅ Conectado ao banco de dados</p>";
    echo "</div>";

    // 1. Verificar estrutura da tabela usuarios
    echo "<h2>📊 Estrutura da Tabela 'usuarios'</h2>";

    $stmt = $pdo->query("DESCRIBE usuarios");
    $columns = $stmt->fetchAll();

    echo "<table>";
    echo "<tr><th>Campo</th><th>Tipo</th><th>Null</th><th>Key</th><th>Default</th><th>Extra</th></tr>";

    $campos_existentes = [];
    foreach ($columns as $col) {
        $campos_existentes[] = $col['Field'];
        echo "<tr>";
        echo "<td><strong>{$col['Field']}</strong></td>";
        echo "<td>{$col['Type']}</td>";
        echo "<td>{$col['Null']}</td>";
        echo "<td>{$col['Key']}</td>";
        echo "<td>" . ($col['Default'] ?? 'NULL') . "</td>";
        echo "<td>{$col['Extra']}</td>";
        echo "</tr>";
    }
    echo "</table>";

    // 2. Verificar quais campos são obrigatórios no INSERT
    echo "<h2>📝 Campos no INSERT do login.php</h2>";

    $campos_insert = ['nome', 'email', 'senha', 'telefone', 'instagram', 'cidade', 'bairro', 'idade', 'foto_perfil', 'role', 'status', 'ativo', 'data_cadastro'];

    echo "<div class='info'>";
    echo "<p><strong>Campos no INSERT:</strong></p>";
    echo "<pre>" . implode(", ", $campos_insert) . "</pre>";
    echo "</div>";

    // 3. Verificar campos faltantes
    $campos_faltantes = array_diff($campos_insert, $campos_existentes);

    if (!empty($campos_faltantes)) {
        echo "<div class='error'>";
        echo "<h3>❌ PROBLEMA ENCONTRADO!</h3>";
        echo "<p><strong>Campos no INSERT que NÃO existem na tabela:</strong></p>";
        echo "<pre>" . implode(", ", $campos_faltantes) . "</pre>";
        echo "</div>";
    } else {
        echo "<div class='success'>";
        echo "<h3>✅ Todos os campos do INSERT existem na tabela!</h3>";
        echo "</div>";
    }

    // 4. Simular INSERT com dados de teste
    echo "<h2>🧪 Testar INSERT</h2>";

    if (isset($_POST['testar_insert'])) {
        try {
            $nome = "Usuário Teste";
            $email = "teste_" . time() . "@example.com";
            $senha = password_hash("senha123", PASSWORD_DEFAULT);
            $telefone = "(11) 99999-9999";
            $instagram = "@teste";
            $cidade = "São Paulo";
            $bairro = "Centro";
            $idade = 25;
            $foto_perfil = "";

            echo "<div class='info'>";
            echo "<p>Tentando inserir usuário de teste...</p>";
            echo "<p><strong>E-mail:</strong> $email</p>";
            echo "</div>";

            $stmt = $pdo->prepare("INSERT INTO usuarios (nome, email, senha, telefone, instagram, cidade, bairro, idade, foto_perfil, role, status, ativo, data_cadastro) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'promoter', 'pendente', 1, NOW())");

            if ($stmt->execute([$nome, $email, $senha, $telefone, $instagram, $cidade, $bairro, $idade, $foto_perfil])) {
                echo "<div class='success'>";
                echo "<h3>🎉 INSERT funcionou!</h3>";
                echo "<p>Usuário de teste criado com sucesso: <strong>$email</strong></p>";
                echo "<p>ID do usuário: " . $pdo->lastInsertId() . "</p>";
                echo "</div>";
            } else {
                echo "<div class='error'>";
                echo "<h3>❌ INSERT falhou</h3>";
                echo "<p>Não foi possível inserir o usuário</p>";
                echo "</div>";
            }

        } catch (PDOException $e) {
            echo "<div class='error'>";
            echo "<h3>❌ ERRO SQL</h3>";
            echo "<p><strong>Mensagem:</strong> " . htmlspecialchars($e->getMessage()) . "</p>";
            echo "<p><strong>Código:</strong> " . $e->getCode() . "</p>";
            echo "</div>";
        }
    }

    echo "<form method='POST'>";
    echo "<button type='submit' name='testar_insert' style='background: #28a745; color: white; padding: 15px 30px; border: none; border-radius: 8px; font-size: 16px; cursor: pointer;'>🧪 TESTAR INSERT AGORA</button>";
    echo "</form>";

    // 5. Verificar se há campos NOT NULL sem default
    echo "<h2>⚠️ Campos Obrigatórios (NOT NULL sem Default)</h2>";

    $campos_obrigatorios = array_filter($columns, function($col) {
        return $col['Null'] === 'NO' &&
               $col['Default'] === null &&
               $col['Extra'] !== 'auto_increment';
    });

    if (!empty($campos_obrigatorios)) {
        echo "<div class='info'>";
        echo "<p>Estes campos <strong>DEVEM</strong> ser preenchidos no INSERT:</p>";
        echo "<table>";
        echo "<tr><th>Campo</th><th>Tipo</th></tr>";
        foreach ($campos_obrigatorios as $col) {
            $em_insert = in_array($col['Field'], $campos_insert);
            $cor = $em_insert ? 'green' : 'red';
            echo "<tr>";
            echo "<td style='color: $cor;'><strong>{$col['Field']}</strong></td>";
            echo "<td>{$col['Type']}</td>";
            echo "</tr>";
        }
        echo "</table>";
        echo "</div>";
    } else {
        echo "<div class='success'>";
        echo "<p>✅ Nenhum campo obrigatório encontrado (ou todos têm default)</p>";
        echo "</div>";
    }

} catch (PDOException $e) {
    echo "<div class='error'>";
    echo "<h3>❌ Erro de Conexão</h3>";
    echo "<p>" . htmlspecialchars($e->getMessage()) . "</p>";
    echo "</div>";
}
?>

</div>
</body>
</html>
