<?php
// debug_login.php - Encontrar erro 500 no login
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <title>Debug Login - Erro 500</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 20px;
            min-height: 100vh;
        }
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            border-radius: 20px;
            padding: 40px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
        }
        h1 { color: #667eea; margin-bottom: 20px; }
        h2 {
            color: #764ba2;
            margin: 25px 0 15px 0;
            padding-bottom: 10px;
            border-bottom: 2px solid #667eea;
        }
        .success {
            background: #d4edda;
            border-left: 4px solid #28a745;
            padding: 15px;
            margin: 15px 0;
            color: #155724;
            border-radius: 5px;
        }
        .error {
            background: #f8d7da;
            border-left: 4px solid #dc3545;
            padding: 15px;
            margin: 15px 0;
            color: #721c24;
            border-radius: 5px;
        }
        .warning {
            background: #fff3cd;
            border-left: 4px solid #ffc107;
            padding: 15px;
            margin: 15px 0;
            color: #856404;
            border-radius: 5px;
        }
        .info {
            background: #d1ecf1;
            border-left: 4px solid #0c5460;
            padding: 15px;
            margin: 15px 0;
            color: #0c5460;
            border-radius: 5px;
        }
        pre {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 5px;
            overflow-x: auto;
            font-size: 12px;
            border: 1px solid #dee2e6;
        }
        code {
            background: #e9ecef;
            padding: 2px 6px;
            border-radius: 3px;
            font-family: 'Courier New', monospace;
        }
        .btn {
            background: #667eea;
            color: white;
            padding: 12px 25px;
            border: none;
            border-radius: 8px;
            font-size: 14px;
            cursor: pointer;
            text-decoration: none;
            display: inline-block;
            margin: 5px;
        }
        .btn:hover { background: #5568d3; }
    </style>
</head>
<body>
<div class="container">
    <h1>🔍 Debug do Erro 500 - Login</h1>

<?php
echo "<h2>📂 Verificar Arquivos</h2>";

$arquivos_verificar = [
    'login.php',
    'config.php',
    'dashboard.php',
    'assets/header.php',
    'assets/footer.php',
    '.htaccess'
];

echo "<table style='width: 100%; border-collapse: collapse; margin: 15px 0;'>";
echo "<tr style='background: #667eea; color: white;'>";
echo "<th style='padding: 12px; text-align: left;'>Arquivo</th>";
echo "<th style='padding: 12px; text-align: left;'>Existe?</th>";
echo "<th style='padding: 12px; text-align: left;'>Tamanho</th>";
echo "<th style='padding: 12px; text-align: left;'>Modificado</th>";
echo "</tr>";

foreach ($arquivos_verificar as $arquivo) {
    $caminho = __DIR__ . '/' . $arquivo;
    $existe = file_exists($caminho);

    echo "<tr style='border-bottom: 1px solid #dee2e6;'>";
    echo "<td style='padding: 10px;'><strong>$arquivo</strong></td>";

    if ($existe) {
        $tamanho = filesize($caminho);
        $modificado = date('d/m/Y H:i:s', filemtime($caminho));
        echo "<td style='padding: 10px; color: green;'>✅ SIM</td>";
        echo "<td style='padding: 10px;'>" . number_format($tamanho) . " bytes</td>";
        echo "<td style='padding: 10px;'>$modificado</td>";
    } else {
        echo "<td style='padding: 10px; color: red;'>❌ NÃO</td>";
        echo "<td style='padding: 10px;'>-</td>";
        echo "<td style='padding: 10px;'>-</td>";
    }

    echo "</tr>";
}

echo "</table>";

// Testar config.php
echo "<h2>🔧 Testar config.php</h2>";

try {
    ob_start();
    require_once __DIR__ . '/config.php';
    $output = ob_get_clean();

    echo "<div class='success'>";
    echo "<p>✅ config.php carregado com sucesso!</p>";

    if (isset($pdo)) {
        echo "<p>✅ Conexão PDO estabelecida!</p>";

        try {
            $stmt = $pdo->query("SELECT COUNT(*) as total FROM usuarios");
            $result = $stmt->fetch();
            echo "<p>✅ Query de teste funcionou! Total de usuários: {$result['total']}</p>";
        } catch (PDOException $e) {
            echo "<p style='color: orange;'>⚠️ Erro na query de teste: " . htmlspecialchars($e->getMessage()) . "</p>";
        }
    } else {
        echo "<p style='color: orange;'>⚠️ Variável \$pdo não foi definida</p>";
    }

    echo "</div>";

} catch (Exception $e) {
    echo "<div class='error'>";
    echo "<h3>❌ ERRO no config.php</h3>";
    echo "<p><strong>Mensagem:</strong> " . htmlspecialchars($e->getMessage()) . "</p>";
    echo "<p><strong>Arquivo:</strong> " . htmlspecialchars($e->getFile()) . "</p>";
    echo "<p><strong>Linha:</strong> " . $e->getLine() . "</p>";
    echo "<pre>" . htmlspecialchars($e->getTraceAsString()) . "</pre>";
    echo "</div>";
}

// Testar login.php (apenas ler, não executar)
echo "<h2>📄 Analisar login.php</h2>";

$login_file = __DIR__ . '/login.php';
if (file_exists($login_file)) {
    echo "<div class='info'>";
    echo "<p>Arquivo existe: <code>$login_file</code></p>";
    echo "<p>Tamanho: " . filesize($login_file) . " bytes</p>";
    echo "</div>";

    // Ler primeiras 50 linhas
    $linhas = file($login_file);
    echo "<div class='warning'>";
    echo "<h3>Primeiras 30 linhas de login.php:</h3>";
    echo "<pre>";
    for ($i = 0; $i < min(30, count($linhas)); $i++) {
        $num = $i + 1;
        echo sprintf("%3d: ", $num) . htmlspecialchars($linhas[$i]);
    }
    echo "</pre>";
    echo "</div>";

    // Verificar sintaxe PHP (sem exec)
    echo "<h3>🔍 Buscar Erros Comuns</h3>";

    $conteudo_login = file_get_contents($login_file);

    $erros_potenciais = [];

    // Verificar PHP_STATUS_NONE
    if (strpos($conteudo_login, 'PHP_STATUS_NONE') !== false) {
        $erros_potenciais[] = "❌ Encontrado 'PHP_STATUS_NONE' - deve ser 'PHP_SESSION_NONE'";
    }

    // Verificar require/include sem tratamento de erro
    if (preg_match('/require.*config\.php/', $conteudo_login)) {
        $erros_potenciais[] = "⚠️ Faz require do config.php - se config.php tiver erro, login.php também terá";
    }

    // Verificar se fecha tags PHP
    $abertura_php = substr_count($conteudo_login, '<?php');
    $fechamento_php = substr_count($conteudo_login, '?>');

    if ($abertura_php > 0 && $fechamento_php > 0 && $abertura_php != $fechamento_php) {
        $erros_potenciais[] = "⚠️ Tags PHP não balanceadas (aberturas: $abertura_php, fechamentos: $fechamento_php)";
    }

    if (!empty($erros_potenciais)) {
        echo "<div class='warning'>";
        echo "<h4>Possíveis problemas encontrados:</h4>";
        echo "<ul>";
        foreach ($erros_potenciais as $erro) {
            echo "<li>$erro</li>";
        }
        echo "</ul>";
        echo "</div>";
    } else {
        echo "<div class='success'>";
        echo "<p>✅ Nenhum erro óbvio detectado no código</p>";
        echo "</div>";
    }

} else {
    echo "<div class='error'>";
    echo "<p>❌ Arquivo login.php não encontrado!</p>";
    echo "</div>";
}

// Testar .htaccess
echo "<h2>⚙️ Verificar .htaccess</h2>";

$htaccess_file = __DIR__ . '/.htaccess';
if (file_exists($htaccess_file)) {
    $htaccess_content = file_get_contents($htaccess_file);
    $linhas_htaccess = count(file($htaccess_file));

    echo "<div class='info'>";
    echo "<p>✅ .htaccess encontrado</p>";
    echo "<p><strong>Linhas:</strong> $linhas_htaccess</p>";
    echo "<p><strong>Tamanho:</strong> " . filesize($htaccess_file) . " bytes</p>";
    echo "</div>";

    echo "<div class='warning'>";
    echo "<h3>Conteúdo do .htaccess:</h3>";
    echo "<pre>" . htmlspecialchars($htaccess_content) . "</pre>";
    echo "</div>";

    // Verificar se tem regras problemáticas
    $problemas_htaccess = [];

    if (strpos($htaccess_content, 'php_value') !== false) {
        $problemas_htaccess[] = "Contém 'php_value' - pode causar erro 500";
    }
    if (strpos($htaccess_content, 'php_flag') !== false) {
        $problemas_htaccess[] = "Contém 'php_flag' - pode causar erro 500";
    }
    if (strpos($htaccess_content, 'RewriteEngine On') !== false && $linhas_htaccess > 50) {
        $problemas_htaccess[] = "Regras de rewrite muito complexas";
    }

    if (!empty($problemas_htaccess)) {
        echo "<div class='error'>";
        echo "<h3>⚠️ Possíveis Problemas no .htaccess:</h3>";
        echo "<ul>";
        foreach ($problemas_htaccess as $problema) {
            echo "<li>$problema</li>";
        }
        echo "</ul>";
        echo "<p><strong>Recomendação:</strong> Renomeie .htaccess para _htaccess.disabled e teste novamente</p>";
        echo "</div>";
    } else {
        echo "<div class='success'>";
        echo "<p>✅ .htaccess parece OK</p>";
        echo "</div>";
    }

} else {
    echo "<div class='warning'>";
    echo "<p>⚠️ Arquivo .htaccess não encontrado (isso pode ser OK)</p>";
    echo "</div>";
}

// Logs do PHP
echo "<h2>📋 Verificar Logs de Erro</h2>";

$possible_log_locations = [
    ini_get('error_log'),
    __DIR__ . '/error_log',
    __DIR__ . '/logs/error.log',
    __DIR__ . '/logs/erros.log',
    '/var/log/apache2/error.log',
    '/var/log/httpd/error_log'
];

echo "<p>Locais verificados para logs:</p>";
echo "<ul>";

$log_encontrado = false;
foreach ($possible_log_locations as $log_path) {
    if ($log_path && file_exists($log_path)) {
        echo "<li style='color: green;'>✅ <code>$log_path</code></li>";
        $log_encontrado = true;

        // Ler últimas 20 linhas
        $log_content = file($log_path);
        $ultimas_linhas = array_slice($log_content, -20);

        echo "<div class='warning'>";
        echo "<h3>Últimas 20 linhas de: $log_path</h3>";
        echo "<pre>";
        foreach ($ultimas_linhas as $linha) {
            echo htmlspecialchars($linha);
        }
        echo "</pre>";
        echo "</div>";

    } else {
        echo "<li style='color: gray;'>❌ <code>$log_path</code></li>";
    }
}
echo "</ul>";

if (!$log_encontrado) {
    echo "<div class='info'>";
    echo "<p>Nenhum log encontrado nos locais padrão.</p>";
    echo "</div>";
}

// Teste final - tentar simular login
echo "<h2>🧪 Teste de Login Simulado</h2>";

echo "<div class='warning'>";
echo "<p>Tente fazer login manualmente:</p>";
echo "<form method='POST' action='login.php' style='background: #f8f9fa; padding: 20px; border-radius: 10px;'>";
echo "<div style='margin-bottom: 15px;'>";
echo "<label style='display: block; margin-bottom: 5px;'><strong>E-mail:</strong></label>";
echo "<input type='email' name='email' required style='width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 5px;'>";
echo "</div>";
echo "<div style='margin-bottom: 15px;'>";
echo "<label style='display: block; margin-bottom: 5px;'><strong>Senha:</strong></label>";
echo "<input type='password' name='senha' required style='width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 5px;'>";
echo "</div>";
echo "<button type='submit' name='login' style='background: #28a745; color: white; padding: 12px 25px; border: none; border-radius: 8px; cursor: pointer;'>🔐 Tentar Login</button>";
echo "</form>";
echo "<p style='margin-top: 15px;'><strong>Se der erro 500:</strong> O erro estará nos logs acima ou no código do login.php</p>";
echo "</div>";

// Recomendações
echo "<h2>💡 Próximos Passos</h2>";

echo "<div class='info'>";
echo "<h3>Baseado no diagnóstico:</h3>";
echo "<ol style='padding-left: 30px; margin: 15px 0;'>";
echo "<li style='margin: 10px 0;'>Se <strong>config.php deu erro</strong>: Corrija o erro mostrado acima</li>";
echo "<li style='margin: 10px 0;'>Se <strong>login.php tem erro de sintaxe</strong>: Corrija a linha indicada</li>";
echo "<li style='margin: 10px 0;'>Se <strong>.htaccess tem problemas</strong>: Renomeie para _htaccess.disabled</li>";
echo "<li style='margin: 10px 0;'>Se <strong>logs mostram erro</strong>: Leia a mensagem e corrija</li>";
echo "<li style='margin: 10px 0;'>Se <strong>tudo parece OK</strong>: Tente o login no formulário acima e veja o erro</li>";
echo "</ol>";
echo "</div>";

?>

</div>
</body>
</html>
