<?php
// financeiro.php - PADRONIZADO COM DESIGN DA DASHBOARD
header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
header("Cache-Control: post-check=0, pre-check=0", false);
header("Pragma: no-cache");

require_once 'config.php';
verificarLogin();

$user_id = $_SESSION['user_id'];
$sucesso = '';
$erro = '';
$mostrar_popup = false;
$valor_popup = 0;

// Solicitar saque PIX
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['solicitar_pix'])) {
    $valor = floatval($_POST['valor_pix']);
    $pix = trim($_POST['pix']);
    $cpf = trim($_POST['cpf']);
    
    // Verificar saldo disponível
    $stmt = $pdo->prepare("SELECT COALESCE(SUM(valor), 0) as saldo FROM comissoes WHERE user_id = ? AND status = 'disponivel'");
    $stmt->execute([$user_id]);
    $saldo = $stmt->fetch()['saldo'];
    
    if ($valor <= 0) {
        $erro = 'Valor inválido.';
    } elseif ($valor > $saldo) {
        $erro = 'Saldo insuficiente. Disponível: ' . formatarMoeda($saldo);
    } elseif (empty($pix)) {
        $erro = 'Informe a chave PIX.';
    } elseif (empty($cpf)) {
        $erro = 'Informe o CPF.';
    } else {
        // Criar solicitação de saque (STATUS: PENDENTE)
        $stmt = $pdo->prepare("INSERT INTO saques (user_id, valor, tipo, pix, cpf, status) VALUES (?, ?, 'pix', ?, ?, 'pendente')");
        if ($stmt->execute([$user_id, $valor, $pix, $cpf])) {
            // Atualizar comissões para 'solicitado'
            $stmt = $pdo->prepare("UPDATE comissoes SET status = 'solicitado' WHERE user_id = ? AND status = 'disponivel'");
            $stmt->execute([$user_id]);
            
            $sucesso = 'Saque solicitado com sucesso! Aguarde a aprovação do administrador.';
        } else {
            $erro = 'Erro ao solicitar saque.';
        }
    }
}

// Solicitar crédito de consumo
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['solicitar_credito'])) {
    $valor = floatval($_POST['valor_credito']);
    
    // Verificar saldo disponível
    $stmt = $pdo->prepare("SELECT COALESCE(SUM(valor), 0) as saldo FROM comissoes WHERE user_id = ? AND status = 'disponivel'");
    $stmt->execute([$user_id]);
    $saldo = $stmt->fetch()['saldo'];
    
    if ($valor <= 0) {
        $erro = 'Valor inválido.';
    } elseif ($valor > $saldo) {
        $erro = 'Saldo insuficiente. Disponível: ' . formatarMoeda($saldo);
    } else {
        // Criar solicitação de crédito (STATUS: PENDENTE)
        $stmt = $pdo->prepare("INSERT INTO saques (user_id, valor, tipo, status) VALUES (?, ?, 'credito', 'pendente')");
        if ($stmt->execute([$user_id, $valor])) {
            // Deduzir do saldo disponível
            $stmt = $pdo->prepare("UPDATE comissoes SET status = 'solicitado' WHERE user_id = ? AND status = 'disponivel'");
            $stmt->execute([$user_id]);
            
            // Mostrar popup
            $mostrar_popup = true;
            $valor_popup = $valor;
        } else {
            $erro = 'Erro ao solicitar crédito.';
        }
    }
}

// Solicitar cartão de consumo usando crédito já aprovado
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['solicitar_cartao_consumo'])) {
    $valor = floatval($_POST['valor_cartao']);

    // Verificar crédito de consumação disponível
    // IMPORTANTE: Não subtrair cartões 'concluido' porque os créditos já foram marcados como 'usado'
    $stmt = $pdo->prepare("
        SELECT COALESCE(SUM(valor), 0) as total
        FROM saques
        WHERE user_id = ? AND tipo = 'credito' AND status IN ('processando', 'concluido')
    ");
    $stmt->execute([$user_id]);
    $credito_total = $stmt->fetch()['total'];

    // Subtrair APENAS cartões pendentes ou processando (ainda não pagos)
    $stmt = $pdo->prepare("
        SELECT COALESCE(SUM(valor), 0) as total
        FROM saques
        WHERE user_id = ? AND tipo = 'cartao_consumo' AND status IN ('pendente', 'processando')
    ");
    $stmt->execute([$user_id]);
    $cartoes_pendentes = $stmt->fetch()['total'];

    $credito_disponivel = $credito_total - $cartoes_pendentes;

    if ($valor <= 0) {
        $erro = 'Valor inválido.';
    } elseif ($valor > $credito_disponivel) {
        $erro = 'Crédito de consumação insuficiente. Disponível: ' . formatarMoeda($credito_disponivel);
    } else {
        try {
            // Criar solicitação de cartão de consumo (STATUS: PENDENTE)
            $stmt = $pdo->prepare("INSERT INTO saques (user_id, valor, tipo, status) VALUES (?, ?, 'cartao_consumo', 'pendente')");
            if ($stmt->execute([$user_id, $valor])) {
                $sucesso = 'Solicitação de cartão de consumo enviada! Aguarde a aprovação do administrador para retirar na bilheteria.';
            } else {
                $errorInfo = $stmt->errorInfo();
                $erro = 'Erro ao solicitar cartão: ' . ($errorInfo[2] ?? 'Erro desconhecido');
                error_log("Erro ao inserir saque cartao_consumo: " . print_r($errorInfo, true));
            }
        } catch (PDOException $e) {
            $erro = 'Erro ao solicitar cartão: ' . $e->getMessage();
            error_log("Exceção PDO ao inserir saque cartao_consumo: " . $e->getMessage());
        }
    }
}

// Buscar dados do usuário para navbar
$stmt = $pdo->prepare("SELECT * FROM usuarios WHERE id = ?");
$stmt->execute([$user_id]);
$usuario = $stmt->fetch();

// Buscar saldo e comissões
$stmt = $pdo->prepare("SELECT COALESCE(SUM(valor), 0) as saldo FROM comissoes WHERE user_id = ? AND status = 'disponivel'");
$stmt->execute([$user_id]);
$saldo_disponivel = $stmt->fetch()['saldo'];

$stmt = $pdo->prepare("SELECT COALESCE(SUM(valor), 0) as total FROM comissoes WHERE user_id = ? AND status = 'solicitado'");
$stmt->execute([$user_id]);
$saldo_solicitado = $stmt->fetch()['total'];

$stmt = $pdo->prepare("SELECT COALESCE(SUM(valor), 0) as total FROM comissoes WHERE user_id = ? AND status = 'pago'");
$stmt->execute([$user_id]);
$total_recebido = $stmt->fetch()['total'];

$stmt = $pdo->prepare("SELECT COALESCE(SUM(valor), 0) as total FROM comissoes WHERE user_id = ?");
$stmt->execute([$user_id]);
$total_geral = $stmt->fetch()['total'];

// Buscar créditos de consumação disponíveis
// IMPORTANTE: Não subtrair cartões 'concluido' porque os créditos já foram marcados como 'usado'
try {
    // Total de créditos aprovados (não incluí 'usado' porque já foram consumidos)
    $stmt = $pdo->prepare("
        SELECT COALESCE(SUM(valor), 0) as total
        FROM saques
        WHERE user_id = ? AND tipo = 'credito' AND status IN ('processando', 'concluido')
    ");
    $stmt->execute([$user_id]);
    $credito_total = $stmt->fetch()['total'] ?? 0;

    // Subtrair APENAS cartões pendentes ou processando (ainda não pagos)
    // NÃO subtrair cartões 'concluido' pois os créditos já foram marcados como 'usado'
    $stmt = $pdo->prepare("
        SELECT COALESCE(SUM(valor), 0) as total
        FROM saques
        WHERE user_id = ? AND tipo = 'cartao_consumo' AND status IN ('pendente', 'processando')
    ");
    $stmt->execute([$user_id]);
    $cartoes_pendentes = $stmt->fetch()['total'] ?? 0;

    $credito_consumacao = max(0, $credito_total - $cartoes_pendentes);
} catch (PDOException $e) {
    error_log("Erro ao calcular crédito de consumação: " . $e->getMessage());
    $credito_consumacao = 0;
}

// Histórico de comissões por evento
$stmt = $pdo->prepare("
    SELECT c.*, e.nome as evento_nome, e.data as evento_data
    FROM comissoes c
    LEFT JOIN eventos e ON c.evento_id = e.id
    WHERE c.user_id = ?
    ORDER BY c.created_at DESC
");
$stmt->execute([$user_id]);
$comissoes = $stmt->fetchAll();

// Histórico de saques
$stmt = $pdo->prepare("SELECT * FROM saques WHERE user_id = ? ORDER BY created_at DESC LIMIT 10");
$stmt->execute([$user_id]);
$saques = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="pt-BR" data-theme="dark">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta http-equiv="Cache-Control" content="no-cache, no-store, must-revalidate">
    <meta http-equiv="Pragma" content="no-cache">
    <meta http-equiv="Expires" content="0">
    
    <title>Financeiro - Sistema Promoters</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="assets/dashboard-style.css">

    <style>
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(240px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        .opcoes-saque {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .opcao-card {
            background: var(--bg-secondary);
            border: 3px solid var(--border-color);
            border-radius: 15px;
            padding: 35px 25px;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .opcao-card:hover {
            border-color: var(--color-gold);
            transform: translateY(-5px);
            box-shadow: 0 10px 30px rgba(255, 215, 0, 0.3);
            background: var(--bg-card-hover);
        }
        
        .opcao-icon {
            font-size: 64px;
            margin-bottom: 20px;
        }
        
        .opcao-titulo {
            font-size: 22px;
            font-weight: 700;
            color: var(--color-gold);
            margin-bottom: 12px;
        }
        
        .opcao-desc {
            font-size: 15px;
            color: var(--text-secondary);
            margin-bottom: 15px;
        }
        
        .opcao-prazo {
            font-size: 13px;
            font-weight: 600;
            padding: 10px 20px;
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            border-radius: 25px;
            display: inline-block;
            color: var(--text-primary);
        }

        .form-saque {
            display: none;
            animation: fadeIn 0.3s;
        }
        
        .form-saque.active {
            display: block;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        .aviso {
            background: rgba(255, 215, 0, 0.1);
            border: 2px solid var(--color-gold);
            border-radius: 12px;
            padding: 20px;
            margin: 20px 0;
            font-size: 14px;
            line-height: 1.6;
        }
        
        .aviso strong {
            color: var(--color-gold);
        }
        .popup-overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.85);
            z-index: 2000;
            align-items: center;
            justify-content: center;
            backdrop-filter: blur(5px);
        }
        
        .popup-overlay.active {
            display: flex;
        }
        
        .popup-content {
            background: var(--bg-card);
            border: 2px solid var(--border-gold);
            border-radius: 20px;
            padding: 50px;
            max-width: 550px;
            width: 90%;
            text-align: center;
            animation: popupIn 0.4s cubic-bezier(0.68, -0.55, 0.265, 1.55);
            box-shadow: 0 20px 60px rgba(0,0,0,0.5);
        }
        
        @keyframes popupIn {
            from { opacity: 0; transform: scale(0.7); }
            to { opacity: 1; transform: scale(1); }
        }
        
        .popup-icon {
            font-size: 90px;
            margin-bottom: 25px;
        }
        
        .popup-content h2 {
            font-size: 28px;
            color: var(--color-gold);
            margin-bottom: 15px;
        }
        
        .popup-content p {
            font-size: 16px;
            color: var(--text-secondary);
            margin-bottom: 25px;
        }
        
        .popup-valor {
            font-size: 56px;
            font-weight: 700;
            color: var(--color-gold);
            margin: 30px 0;
        }
        
        @media (max-width: 768px) {
            body {
                padding: 10px;
            }

            .welcome-section {
                padding: 20px;
                margin: 0 -10px 20px;
            }

            .stats-grid {
                grid-template-columns: 1fr;
                gap: 15px;
            }

            .stat-card {
                padding: 20px;
            }

            .opcoes-saque {
                grid-template-columns: 1fr;
                gap: 15px;
            }

            .saque-card {
                padding: 20px;
            }

            .popup-content {
                padding: 20px;
                margin: 10px;
                max-width: calc(100% - 20px);
            }

            .form-group input,
            .form-group select {
                font-size: 16px;
            }

            .card {
                padding: 20px;
                margin: 0 0 20px;
            }

            table {
                font-size: 13px;
            }

            table td,
            table th {
                padding: 10px 8px;
            }
        }
    </style>
</head>
<body>

    <?php include 'assets/sidebar-navbar.php'; ?>

    <div class="main-content">
        <div class="content-wrapper">
            <div class="container">

        <div class="welcome-section">
            <h1 class="welcome-title">
                <i class="fas fa-wallet"></i>
                Gestão Financeira
            </h1>
            <p class="welcome-subtitle">
                Gerencie suas comissões e solicite saques
            </p>
        </div>

        <?php if ($sucesso): ?>
            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i>
                <?= htmlspecialchars($sucesso) ?>
            </div>
        <?php endif; ?>
        
        <?php if ($erro): ?>
            <div class="alert alert-error">
                <i class="fas fa-exclamation-circle"></i>
                <?= htmlspecialchars($erro) ?>
            </div>
        <?php endif; ?>

        <div class="stats-grid">
            <div class="stat-card destaque">
                <i class="fas fa-wallet stat-icon"></i>
                <div class="stat-value"><?= formatarMoeda($saldo_disponivel) ?></div>
                <div class="stat-label">Saldo Disponível</div>
            </div>
            
            <div class="stat-card">
                <i class="fas fa-clock stat-icon"></i>
                <div class="stat-value"><?= formatarMoeda($saldo_solicitado) ?></div>
                <div class="stat-label">Saldo Solicitado</div>
            </div>
            
            <div class="stat-card">
                <i class="fas fa-check-circle stat-icon"></i>
                <div class="stat-value"><?= formatarMoeda($total_recebido) ?></div>
                <div class="stat-label">Total Recebido</div>
            </div>
              
            <div class="stat-card">
                <i class="fas fa-glass-cheers stat-icon"></i>
                <div class="stat-value"><?= formatarMoeda($credito_consumacao ?? 0) ?></div>
                <div class="stat-label">Crédito Consumação</div>
            </div>
        </div>
        
        <?php if ($saldo_disponivel > 0): ?>
        <div class="card">
            <div class="card-title">
                <i class="fas fa-hand-holding-usd"></i>
                Escolha como deseja receber
            </div>

            <div class="opcoes-saque">
                <div class="opcao-card" onclick="mostrarFormulario('pix')">
                    <div class="opcao-icon"><i class="fas fa-pix" style="font-size: 64px; color: var(--color-gold);"></i></div>
                    <div class="opcao-titulo">PIX</div>
                    <div class="opcao-desc">Transferência para sua conta</div>
                    <div class="opcao-prazo">3 a 5 dias úteis após o evento</div>
                </div>

                <div class="opcao-card" onclick="mostrarFormulario('credito')">
                    <div class="opcao-icon"><i class="fas fa-glass-cheers" style="font-size: 64px; color: var(--color-gold);"></i></div>
                    <div class="opcao-titulo">Crédito de Consumo</div>
                    <div class="opcao-desc">Use na bilheteria do evento</div>
                    <div class="opcao-prazo">Disponível imediatamente</div>
                </div>
            </div>
            
            <div id="form-pix" class="form-saque">
                <h3 style="color: var(--color-gold); margin-bottom: 25px; font-size: 22px;">
                    <i class="fas fa-pix"></i> Saque via PIX
                </h3>

                <div class="aviso">
                    <strong><i class="fas fa-exclamation-triangle"></i> IMPORTANTE:</strong> O valor será creditado em sua conta corrente via PIX em <strong>3 a 5 dias úteis após o evento</strong>.
                </div>
                
                <form method="POST">
                    <div class="form-group">
                        <label><i class="fas fa-money-bill-wave"></i> Valor do Saque</label>
                        <input type="number" name="valor_pix" step="0.01" min="0.01" max="<?= $saldo_disponivel ?>" required placeholder="R$ 0,00">
                        <small>Disponível: <?= formatarMoeda($saldo_disponivel) ?></small>
                    </div>
                    
                    <div class="form-group">
                        <label><i class="fas fa-key"></i> Chave PIX</label>
                        <input type="text" name="pix" required placeholder="email@exemplo.com, CPF ou telefone">
                    </div>
                    
                    <div class="form-group">
                        <label><i class="fas fa-id-card"></i> CPF</label>
                        <input type="text" name="cpf" required placeholder="000.000.000-00">
                    </div>
                    
                    <button type="submit" name="solicitar_pix" class="btn btn-primary">
                        <i class="fas fa-paper-plane"></i>
                        Solicitar Saque via PIX
                    </button>
                </form>
            </div>
            
            <div id="form-credito" class="form-saque">
                <h3 style="color: var(--color-gold); margin-bottom: 25px; font-size: 22px;">
                    <i class="fas fa-glass-martini-alt"></i> Crédito de Consumo
                </h3>

                <div class="aviso">
                    <strong><i class="fas fa-check-circle"></i> ATENÇÃO:</strong> Após confirmar, dirija-se à <strong>bilheteria do evento</strong> para retirar seu crédito de consumo.
                </div>
                
                <form method="POST">
                    <div class="form-group">
                        <label><i class="fas fa-money-bill-wave"></i> Valor do Crédito</label>
                        <input type="number" name="valor_credito" step="0.01" min="0.01" max="<?= $saldo_disponivel ?>" required placeholder="R$ 0,00">
                        <small>Disponível: <?= formatarMoeda($saldo_disponivel) ?></small>
                    </div>
                    
                    <button type="submit" name="solicitar_credito" class="btn btn-primary">
                        <i class="fas fa-ticket-alt"></i>
                        Gerar Crédito de Consumo
                    </button>
                </form>
            </div>
        </div>
        <?php else: ?>
        <div class="card">
            <div class="empty-state">
                <div class="empty-state-icon">
                    <i class="fas fa-wallet"></i>
                </div>
                <h3>Saldo Insuficiente</h3>
                <p>Continue divulgando eventos para gerar comissões!</p>
            </div>
        </div>
        <?php endif; ?>

        <?php if ($credito_consumacao > 0): ?>
        <div class="card">
            <div class="card-title">
                <i class="fas fa-credit-card"></i>
                Solicitar Cartão de Consumo
            </div>

            <div class="aviso" style="background: rgba(255, 165, 0, 0.1); border-left: 4px solid #FFA500;">
                <strong><i class="fas fa-info-circle"></i> CRÉDITO DE CONSUMAÇÃO:</strong> Você possui <strong><?= formatarMoeda($credito_consumacao) ?></strong> em crédito de consumação (BÔNUS).
                Este valor NÃO pode ser convertido em dinheiro. Solicite um cartão de consumo para usar na bilheteria do evento.
            </div>

            <div class="opcoes-saque">
                <div class="opcao-card" onclick="mostrarFormulario('cartao_consumo')">
                    <div class="opcao-icon"><i class="fas fa-credit-card" style="font-size: 64px; color: #FFA500;"></i></div>
                    <div class="opcao-titulo">Cartão de Consumo</div>
                    <div class="opcao-desc">Use seu bônus na bilheteria</div>
                    <div class="opcao-prazo">Aguarda aprovação do admin</div>
                </div>
            </div>

            <div id="form-cartao_consumo" class="form-saque">
                <h3 style="color: #FFA500; margin-bottom: 25px; font-size: 22px;">
                    <i class="fas fa-credit-card"></i> Solicitar Cartão de Consumo
                </h3>

                <div class="aviso">
                    <strong><i class="fas fa-exclamation-triangle"></i> IMPORTANTE:</strong> Este é um <strong>BÔNUS</strong> e não pode ser convertido em dinheiro.
                    Após aprovação do administrador, você poderá retirar seu cartão de consumo na bilheteria do evento.
                </div>

                <form method="POST">
                    <div class="form-group">
                        <label><i class="fas fa-money-bill-wave"></i> Valor do Cartão</label>
                        <input type="number" name="valor_cartao" step="0.01" min="0.01" max="<?= $credito_consumacao ?>" required placeholder="R$ 0,00">
                        <small>Crédito disponível: <?= formatarMoeda($credito_consumacao) ?></small>
                    </div>

                    <button type="submit" name="solicitar_cartao_consumo" class="btn btn-primary" style="background: linear-gradient(135deg, #FFA500 0%, #FF8C00 100%);">
                        <i class="fas fa-paper-plane"></i>
                        Solicitar Cartão de Consumo
                    </button>
                </form>
            </div>
        </div>
        <?php endif; ?>
        
        <div class="card">
            <div class="card-title">
                <i class="fas fa-chart-bar"></i>
                Comissões por Evento
            </div>
            
            <?php if (count($comissoes) > 0): ?>
                <div style="overflow-x: auto;">
                    <table>
                        <thead>
                            <tr>
                                <th>Evento</th>
                                <th>Data</th>
                                <th>Convidados</th>
                                <th>Check-ins</th>
                                <th>Valor</th>
                                <th>Status</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($comissoes as $com): ?>
                                <tr>
                                    <td><?= htmlspecialchars($com['evento_nome']) ?></td>
                                    <td><?= formatarData($com['evento_data']) ?></td>
                                    <td><?= $com['convidados_total'] ?></td>
                                    <td><?= $com['convidados_checkin'] ?></td>
                                    <td style="font-weight: bold; color: var(--color-gold);">
                                        <?= formatarMoeda($com['valor']) ?>
                                    </td>
                                    <td>
                                        <?php if ($com['status'] === 'disponivel'): ?>
                                            <span class="badge badge-success">Disponível</span>
                                        <?php elseif ($com['status'] === 'solicitado'): ?>
                                            <span class="badge badge-warning">Solicitado</span>
                                        <?php else: ?>
                                            <span class="badge badge-info">Pago</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php else: ?>
                <div class="empty-state">
                    <div class="empty-state-icon">
                        <i class="fas fa-chart-bar"></i>
                    </div>
                    <h3>Nenhuma Comissão</h3>
                    <p>Comece a divulgar eventos para gerar comissões!</p>
                </div>
            <?php endif; ?>
        </div>
        
        <?php if (count($saques) > 0): ?>
        <div class="card">
            <div class="card-title">
                <i class="fas fa-history"></i>
                Histórico de Saques
            </div>
            
            <div style="overflow-x: auto;">
                <table>
                    <thead>
                        <tr>
                            <th>Data</th>
                            <th>Tipo</th>
                            <th>Valor</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($saques as $saque): ?>
                            <tr>
                                <td><?= formatarDataHora($saque['created_at']) ?></td>
                                <td>
                                    <?php if ($saque['tipo'] === 'pix'): ?>
                                        <i class="fas fa-pix"></i> PIX
                                    <?php elseif ($saque['tipo'] === 'cartao_consumo'): ?>
                                        <i class="fas fa-credit-card"></i> Cartão de Consumo
                                    <?php else: ?>
                                        <i class="fas fa-glass-cheers"></i> Crédito de Consumação
                                    <?php endif; ?>
                                </td>
                                <td style="font-weight: bold; color: var(--color-gold);">
                                    <?= formatarMoeda($saque['valor']) ?>
                                </td>
                                <td>
                                    <?php if ($saque['status'] === 'pendente'): ?>
                                        <span class="badge badge-warning">Pendente</span>
                                    <?php elseif ($saque['status'] === 'processando'): ?>
                                        <span class="badge badge-info">Processando</span>
                                    <?php elseif ($saque['status'] === 'concluido'): ?>
                                        <span class="badge badge-success">Concluído</span>
                                    <?php else: ?>
                                        <span class="badge badge-danger">Cancelado</span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php endif; ?>

            </div><!-- .container -->
        </div><!-- .content-wrapper -->
    </div><!-- .main-content -->

    <div class="popup-overlay <?= $mostrar_popup ? 'active' : '' ?>" id="popupCredito">
        <div class="popup-content">
            <div class="popup-icon"><i class="fas fa-check-circle" style="color: var(--success-color);"></i></div>
            <h2>Crédito Gerado!</h2>
            <p>Seu crédito de consumo foi aprovado:</p>
            <div class="popup-valor"><?= formatarMoeda($valor_popup) ?></div>
            <div class="aviso">
                <strong><i class="fas fa-map-marker-alt"></i> Dirija-se à bilheteria do evento</strong><br>
                Apresente este comprovante e seu documento para retirar o crédito.
            </div>
            <button class="btn btn-secondary" onclick="fecharPopup()">
                <i class="fas fa-times"></i>
                Fechar
            </button>
        </div>
    </div>
    
    <script>
        function mostrarFormulario(tipo) {
            document.querySelectorAll('.form-saque').forEach(form => {
                form.classList.remove('active');
            });
            
            const formElement = document.getElementById('form-' + tipo);
            if (formElement) {
                formElement.classList.add('active');
                
                setTimeout(() => {
                    formElement.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
                }, 100);
            }
        }
        
        function fecharPopup() {
            const popup = document.getElementById('popupCredito');
            if (popup) {
                popup.classList.remove('active');
            }
        }
        
        const popupOverlay = document.getElementById('popupCredito');
        if (popupOverlay) {
            popupOverlay.addEventListener('click', function(e) {
                if (e.target === this) {
                    fecharPopup();
                }
            });
        }
        
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                fecharPopup();
            }
        });
    </script>
</body>
</html>