<?php
// login.php
require_once 'config.php';

$erro = '';
$sucesso = '';

// Buscar configurações do sistema
$stmt = $pdo->query("SELECT * FROM configuracoes WHERE id = 1");
$config = $stmt->fetch();

// Se não existir configuração, usar valores padrão
if (!$config) {
    $config = [
        'nome_sistema' => 'Sistema Promoters',
        'logo_sistema' => '',
        'banner_principal' => '',
        'cor_primaria' => '#FFD700',
        'cor_secundaria' => '#FFA500'
    ];
}

// Processar login
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['login'])) {
    $email = filter_input(INPUT_POST, 'email', FILTER_SANITIZE_EMAIL);
    $senha = $_POST['senha'];
    
    $stmt = $pdo->prepare("SELECT * FROM usuarios WHERE email = ? AND status = 'aprovado'");
    $stmt->execute([$email]);
    $user = $stmt->fetch();
    
    if ($user && password_verify($senha, $user['senha'])) {
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['nome'] = $user['nome'];
        $_SESSION['role'] = $user['role'];
        header('Location: dashboard.php');
        exit;
    } else {
        $erro = 'Email ou senha incorretos, ou conta não aprovada.';
    }
}

// Processar registro
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['registro'])) {
    $nome = htmlspecialchars(trim($_POST['nome'] ?? ''), ENT_QUOTES, 'UTF-8');
    $email = filter_input(INPUT_POST, 'email', FILTER_SANITIZE_EMAIL);
    $senha = password_hash($_POST['senha'], PASSWORD_DEFAULT);
    $telefone = htmlspecialchars(trim($_POST['telefone'] ?? ''), ENT_QUOTES, 'UTF-8');
    $instagram = htmlspecialchars(trim($_POST['instagram'] ?? ''), ENT_QUOTES, 'UTF-8');
    $cidade = htmlspecialchars(trim($_POST['cidade'] ?? ''), ENT_QUOTES, 'UTF-8');
    $bairro = htmlspecialchars(trim($_POST['bairro'] ?? ''), ENT_QUOTES, 'UTF-8');
    $idade = filter_input(INPUT_POST, 'idade', FILTER_SANITIZE_NUMBER_INT);
    
    $foto_perfil = '';
    
    // Upload da foto de perfil
    if (isset($_FILES['foto_perfil']) && $_FILES['foto_perfil']['error'] == 0) {
        $allowed = ['jpg', 'jpeg', 'png', 'gif'];
        $filename = $_FILES['foto_perfil']['name'];
        $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        $filesize = $_FILES['foto_perfil']['size'];
        
        // Validar extensão e tamanho (máx 5MB)
        if (in_array($ext, $allowed) && $filesize <= 5242880) {
            $novo_nome = 'perfil_' . uniqid() . '.' . $ext;
            $destino = 'uploads/perfis/' . $novo_nome;
            
            if (!is_dir('uploads/perfis')) {
                mkdir('uploads/perfis', 0777, true);
            }
            
            if (move_uploaded_file($_FILES['foto_perfil']['tmp_name'], $destino)) {
                $foto_perfil = $destino;
            } else {
                $erro = 'Erro ao fazer upload da foto.';
            }
        } else {
            $erro = 'Formato de imagem inválido ou arquivo muito grande (máx 5MB).';
        }
    }
    
    // Verificar se email já existe
    if (empty($erro)) {
        try {
            $stmt = $pdo->prepare("SELECT id FROM usuarios WHERE email = ?");
            $stmt->execute([$email]);

            if ($stmt->fetch()) {
                $erro = 'Este email já está cadastrado.';
            } else {
                $stmt = $pdo->prepare("INSERT INTO usuarios (nome, email, senha, telefone, instagram, cidade, bairro, idade, foto_perfil, role, status, ativo, data_cadastro) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'promoter', 'pendente', 1, NOW())");
                if ($stmt->execute([$nome, $email, $senha, $telefone, $instagram, $cidade, $bairro, $idade, $foto_perfil])) {
                    $sucesso = 'Cadastro realizado com sucesso! Aguarde aprovação do administrador.';
                } else {
                    $erro = 'Erro ao realizar cadastro. Por favor, tente novamente.';
                }
            }
        } catch (PDOException $e) {
            error_log("Erro no cadastro: " . $e->getMessage());
            $erro = 'Erro ao realizar cadastro: ' . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - <?= htmlspecialchars($config['nome_sistema']) ?></title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root {
            --bg-primary: #000000;
            --bg-secondary: #1a1a1a;
            --color-primary: <?= $config['cor_primaria'] ?>;
            --color-secondary: <?= $config['cor_secundaria'] ?>;
            --color-text: #ffffff;
            --color-error: #ff4444;
            --color-success: #00C851;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, var(--bg-primary) 0%, var(--bg-secondary) 100%);
            color: var(--color-text);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .container {
            width: 100%;
            max-width: 500px;
            background: var(--bg-secondary);
            border-radius: 20px;
            box-shadow: 0 10px 40px rgba(255, 215, 0, 0.1);
            overflow: hidden;
        }
        
        .header {
            background: linear-gradient(135deg, var(--color-primary) 0%, var(--color-secondary) 100%);
            padding: 30px;
            text-align: center;
            color: var(--bg-primary);
            position: relative;
        }

        <?php if (!empty($config['banner_principal'])): ?>
        .header::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: url('<?= htmlspecialchars($config['banner_principal']) ?>') center/cover;
            opacity: 0.2;
        }
        <?php endif; ?>

        .header-content {
            position: relative;
            z-index: 1;
        }

        .logo {
            max-width: 120px;
            max-height: 80px;
            margin-bottom: 15px;
            filter: drop-shadow(0 2px 8px rgba(0,0,0,0.3));
        }
        
        .header h1 {
            font-size: 28px;
            font-weight: bold;
            margin-bottom: 5px;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.2);
        }
        
        .header p {
            font-size: 14px;
            opacity: 0.9;
        }
        
        .tabs {
            display: flex;
            background: var(--bg-primary);
        }
        
        .tab {
            flex: 1;
            padding: 15px;
            text-align: center;
            cursor: pointer;
            border: none;
            background: transparent;
            color: var(--color-text);
            font-size: 16px;
            font-weight: 600;
            transition: all 0.3s;
            border-bottom: 3px solid transparent;
        }
        
        .tab.active {
            border-bottom-color: var(--color-primary);
            color: var(--color-primary);
        }
        
        .form-container {
            padding: 30px;
            max-height: 70vh;
            overflow-y: auto;
        }

        .form-container::-webkit-scrollbar {
            width: 8px;
        }

        .form-container::-webkit-scrollbar-track {
            background: var(--bg-primary);
        }

        .form-container::-webkit-scrollbar-thumb {
            background: var(--color-primary);
            border-radius: 4px;
        }
        
        .form-content {
            display: none;
        }
        
        .form-content.active {
            display: block;
            animation: fadeIn 0.3s;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        label {
            display: block;
            margin-bottom: 8px;
            font-size: 14px;
            font-weight: 500;
            color: var(--color-primary);
        }
        
        input[type="text"],
        input[type="email"],
        input[type="password"],
        input[type="number"] {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #333;
            border-radius: 10px;
            background: var(--bg-primary);
            color: var(--color-text);
            font-size: 14px;
            transition: all 0.3s;
        }
        
        input:focus {
            outline: none;
            border-color: var(--color-primary);
            box-shadow: 0 0 0 3px rgba(255, 215, 0, 0.1);
        }

        .file-input-wrapper {
            position: relative;
            margin-bottom: 20px;
        }

        .file-input-label {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
            padding: 40px 20px;
            background: var(--bg-primary);
            border: 2px dashed #333;
            border-radius: 10px;
            cursor: pointer;
            transition: all 0.3s;
            font-size: 14px;
            color: #999;
        }

        .file-input-label:hover {
            border-color: var(--color-primary);
            background: rgba(255, 215, 0, 0.05);
            color: var(--color-primary);
        }

        .file-input-wrapper input[type="file"] {
            position: absolute;
            opacity: 0;
            width: 0;
            height: 0;
        }

        .preview-container {
            display: none;
            margin-top: 15px;
            text-align: center;
        }

        .preview-container.active {
            display: block;
        }

        .preview-image {
            max-width: 150px;
            max-height: 150px;
            border-radius: 50%;
            border: 3px solid var(--color-primary);
            object-fit: cover;
        }

        .preview-label {
            display: block;
            margin-top: 10px;
            font-size: 12px;
            color: #999;
        }
        
        .btn {
            width: 100%;
            padding: 14px;
            border: none;
            border-radius: 10px;
            background: linear-gradient(135deg, var(--color-primary) 0%, var(--color-secondary) 100%);
            color: var(--bg-primary);
            font-size: 16px;
            font-weight: bold;
            cursor: pointer;
            transition: all 0.3s;
            margin-top: 10px;
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 20px rgba(255, 215, 0, 0.3);
        }
        
        .btn:active {
            transform: translateY(0);
        }
        
        .alert {
            padding: 12px 15px;
            border-radius: 10px;
            margin-bottom: 20px;
            font-size: 14px;
        }
        
        .alert-error {
            background: rgba(255, 68, 68, 0.1);
            border: 1px solid var(--color-error);
            color: var(--color-error);
        }
        
        .alert-success {
            background: rgba(0, 200, 81, 0.1);
            border: 1px solid var(--color-success);
            color: var(--color-success);
        }
        
        .footer {
            text-align: center;
            padding: 20px;
            font-size: 12px;
            color: #999;
            border-top: 1px solid #333;
        }
        
        .footer a {
            color: var(--color-primary);
            text-decoration: none;
        }
        
        .grid-2 {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
        }
        
        @media (max-width: 768px) {
            .grid-2 {
                grid-template-columns: 1fr;
            }
            
            .container {
                max-width: 100%;
            }
            
            .header h1 {
                font-size: 24px;
            }

            .form-container {
                max-height: 60vh;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <div class="header-content">
                <?php if (!empty($config['logo_sistema'])): ?>
                    <img src="<?= htmlspecialchars($config['logo_sistema']) ?>" alt="Logo" class="logo">
                <?php else: ?>
                    <h1><i class="fas fa-theater-masks" style="color: #000;"></i></h1>
                <?php endif; ?>
                <h1><?= htmlspecialchars($config['nome_sistema']) ?></h1>
                <p>Espaço Taal</p>
            </div>
        </div>
        
        <div class="tabs">
            <button class="tab active" onclick="switchTab('login')">Login</button>
            <button class="tab" onclick="switchTab('registro')">Cadastro</button>
        </div>
        
        <div class="form-container">
            <?php if ($erro): ?>
                <div class="alert alert-error"><?= htmlspecialchars($erro) ?></div>
            <?php endif; ?>
            
            <?php if ($sucesso): ?>
                <div class="alert alert-success"><?= htmlspecialchars($sucesso) ?></div>
            <?php endif; ?>
            
            <!-- Formulário de Login -->
            <div id="login" class="form-content active">
                <form method="POST">
                    <div class="form-group">
                        <label>Email</label>
                        <input type="email" name="email" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Senha</label>
                        <input type="password" name="senha" required>
                    </div>
                    
                    <button type="submit" name="login" class="btn">Entrar</button>
                </form>
            </div>
            
            <!-- Formulário de Registro -->
            <div id="registro" class="form-content">
                <form method="POST" enctype="multipart/form-data">
                    <div class="file-input-wrapper">
                        <label class="file-input-label" for="foto_perfil">
                            <i class="fas fa-camera" style="font-size: 32px;"></i>
                            <span>Adicionar Foto de Perfil</span>
                        </label>
                        <input type="file" id="foto_perfil" name="foto_perfil" accept="image/*" onchange="previewImage(event)">
                    </div>

                    <div class="preview-container" id="preview-container">
                        <img id="preview-image" class="preview-image" alt="Preview">
                        <span class="preview-label">Prévia da foto</span>
                    </div>
                    
                    <div class="form-group">
                        <label>Nome Completo</label>
                        <input type="text" name="nome" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Email</label>
                        <input type="email" name="email" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Senha</label>
                        <input type="password" name="senha" required minlength="6">
                    </div>
                    
                    <div class="grid-2">
                        <div class="form-group">
                            <label>Telefone</label>
                            <input type="text" name="telefone" required>
                        </div>
                        
                        <div class="form-group">
                            <label>Instagram</label>
                            <input type="text" name="instagram" placeholder="@usuario">
                        </div>
                    </div>
                    
                    <div class="grid-2">
                        <div class="form-group">
                            <label>Cidade</label>
                            <input type="text" name="cidade" required>
                        </div>
                        
                        <div class="form-group">
                            <label>Bairro</label>
                            <input type="text" name="bairro" required>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Idade</label>
                        <input type="number" name="idade" required min="18" max="100">
                    </div>
                    
                    <button type="submit" name="registro" class="btn">Cadastrar</button>
                </form>
            </div>
        </div>
        
        <div class="footer">
            Desenvolvido por <a href="#">Alerson Araújo</a><br>
            Todos os direitos reservados<br>
            Licenciado para <strong>Espaço Taal</strong>
        </div>
    </div>
    
    <script>
        function switchTab(tab) {
            // Remove active de todas as tabs
            document.querySelectorAll('.tab').forEach(t => t.classList.remove('active'));
            document.querySelectorAll('.form-content').forEach(f => f.classList.remove('active'));
            
            // Ativa a tab selecionada
            event.target.classList.add('active');
            document.getElementById(tab).classList.add('active');
        }

        function previewImage(event) {
            const file = event.target.files[0];
            const previewContainer = document.getElementById('preview-container');
            const previewImage = document.getElementById('preview-image');
            const label = event.target.previousElementSibling;

            if (file) {
                const reader = new FileReader();
                
                reader.onload = function(e) {
                    previewImage.src = e.target.result;
                    previewContainer.classList.add('active');
                    label.innerHTML = '<i class="fas fa-check-circle" style="font-size: 32px; color: #00C851;"></i><span>Foto Selecionada</span>';
                    label.style.borderColor = 'var(--color-primary)';
                    label.style.backgroundColor = 'rgba(255, 215, 0, 0.05)';
                }

                reader.readAsDataURL(file);
            } else {
                previewContainer.classList.remove('active');
                label.innerHTML = '<i class="fas fa-camera" style="font-size: 32px;"></i><span>Adicionar Foto de Perfil</span>';
                label.style.borderColor = '#333';
                label.style.backgroundColor = 'var(--bg-primary)';
            }
        }
    </script>
</body>
</html>