<?php
// produtor_financeiro.php - Financeiro para produtores
header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
header("Cache-Control: post-check=0, pre-check=0", false);
header("Pragma: no-cache");

require_once 'config.php';
verificarLogin();

// Verificar se é produtor
if ($_SESSION['role'] !== 'produtor') {
    header('Location: dashboard.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$sucesso = '';
$erro = '';

// Solicitar saque PIX
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['solicitar_pix'])) {
    $valor = floatval($_POST['valor_pix']);
    $pix = trim($_POST['pix']);
    $cpf = trim($_POST['cpf']);

    // Verificar saldo disponível
    $stmt = $pdo->prepare("SELECT COALESCE(SUM(valor), 0) as saldo FROM comissoes WHERE user_id = ? AND status = 'disponivel'");
    $stmt->execute([$user_id]);
    $saldo = $stmt->fetch()['saldo'];

    if ($valor <= 0) {
        $erro = 'Valor inválido.';
    } elseif ($valor > $saldo) {
        $erro = 'Saldo insuficiente. Disponível: ' . formatarMoeda($saldo);
    } elseif (empty($pix)) {
        $erro = 'Informe a chave PIX.';
    } elseif (empty($cpf)) {
        $erro = 'Informe o CPF.';
    } else {
        // Criar solicitação de saque
        $stmt = $pdo->prepare("INSERT INTO saques (user_id, valor, tipo, pix, cpf, status) VALUES (?, ?, 'pix', ?, ?, 'pendente')");
        if ($stmt->execute([$user_id, $valor, $pix, $cpf])) {
            // Atualizar comissões para 'solicitado'
            $stmt = $pdo->prepare("UPDATE comissoes SET status = 'solicitado' WHERE user_id = ? AND status = 'disponivel'");
            $stmt->execute([$user_id]);

            $sucesso = 'Saque solicitado com sucesso! Aguarde a aprovação do administrador.';
        } else {
            $erro = 'Erro ao solicitar saque.';
        }
    }
}

// Buscar saldo e comissões
$stmt = $pdo->prepare("SELECT COALESCE(SUM(valor), 0) as saldo FROM comissoes WHERE user_id = ? AND status = 'disponivel'");
$stmt->execute([$user_id]);
$saldo_disponivel = $stmt->fetch()['saldo'];

$stmt = $pdo->prepare("SELECT COALESCE(SUM(valor), 0) as total FROM comissoes WHERE user_id = ? AND status = 'solicitado'");
$stmt->execute([$user_id]);
$saldo_solicitado = $stmt->fetch()['total'];

$stmt = $pdo->prepare("SELECT COALESCE(SUM(valor), 0) as total FROM comissoes WHERE user_id = ? AND status = 'pago'");
$stmt->execute([$user_id]);
$total_recebido = $stmt->fetch()['total'];

// Buscar comissões por evento
$stmt = $pdo->prepare("
    SELECT c.*, e.nome as evento_nome, e.data as evento_data
    FROM comissoes c
    INNER JOIN eventos e ON c.evento_id = e.id
    WHERE c.user_id = ?
    ORDER BY e.data DESC
");
$stmt->execute([$user_id]);
$comissoes = $stmt->fetchAll();

// Buscar histórico de saques
$stmt = $pdo->prepare("SELECT * FROM saques WHERE user_id = ? ORDER BY created_at DESC");
$stmt->execute([$user_id]);
$saques = $stmt->fetchAll();

// Buscar dados do usuário para navbar
$stmt = $pdo->prepare("SELECT * FROM usuarios WHERE id = ?");
$stmt->execute([$user_id]);
$usuario = $stmt->fetch();
?>
<!DOCTYPE html>
<html lang="pt-BR" data-theme="dark">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Financeiro - Produtor</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="assets/dashboard-style.css">
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }

        :root {
            --color-gold: #FFD700;
            --color-gold-dark: #FFA500;
            --bg-primary: #000000;
            --bg-secondary: #0a0a0a;
            --bg-card: #1a1a1a;
            --text-primary: #FFFFFF;
            --text-secondary: #999999;
            --border-color: #2a2a2a;
        }

        body {
            font-family: 'Inter', -apple-system, sans-serif;
            background: var(--bg-secondary);
            color: var(--text-primary);
            line-height: 1.6;
        }

        .navbar {
            background: var(--bg-primary);
            border-bottom: 2px solid var(--border-color);
            padding: 20px;
            position: sticky;
            top: 0;
            z-index: 1000;
        }

        .navbar-container {
            max-width: 1400px;
            margin: 0 auto;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .navbar-brand {
            font-size: 28px;
            font-weight: 700;
            color: var(--color-gold);
            display: flex;
            align-items: center;
            gap: 12px;
        }

        .navbar-links {
            display: flex;
            gap: 10px;
        }

        .nav-link {
            color: var(--text-primary);
            text-decoration: none;
            padding: 10px 20px;
            border-radius: 10px;
            transition: all 0.3s;
            font-size: 14px;
            border: 2px solid var(--border-color);
        }

        .nav-link:hover {
            border-color: var(--color-gold);
            color: var(--color-gold);
        }

        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 30px 20px;
        }

        .alert {
            padding: 15px 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }

        .alert-success {
            background: rgba(0, 200, 81, 0.15);
            border: 2px solid #00C851;
            color: #00C851;
        }

        .alert-error {
            background: rgba(255, 68, 68, 0.15);
            border: 2px solid #ff4444;
            color: #ff4444;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: var(--bg-card);
            border: 2px solid var(--border-color);
            border-radius: 15px;
            padding: 25px;
            text-align: center;
        }

        .stat-icon {
            font-size: 40px;
            color: var(--color-gold);
            margin-bottom: 10px;
        }

        .stat-value {
            font-size: 32px;
            font-weight: 700;
            color: var(--color-gold);
            margin-bottom: 5px;
        }

        .stat-label {
            font-size: 12px;
            color: var(--text-secondary);
            text-transform: uppercase;
        }

        .section {
            background: var(--bg-card);
            border: 2px solid var(--border-color);
            border-radius: 15px;
            padding: 30px;
            margin-bottom: 30px;
        }

        .section-title {
            font-size: 24px;
            font-weight: 700;
            margin-bottom: 20px;
            color: var(--color-gold);
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: var(--text-secondary);
        }

        .form-input {
            width: 100%;
            padding: 12px;
            border: 2px solid var(--border-color);
            border-radius: 10px;
            background: var(--bg-secondary);
            color: var(--text-primary);
            font-size: 14px;
        }

        .form-input:focus {
            outline: none;
            border-color: var(--color-gold);
        }

        .btn {
            padding: 12px 25px;
            border: none;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--color-gold) 0%, var(--color-gold-dark) 100%);
            color: #000;
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(255, 215, 0, 0.4);
        }

        table {
            width: 100%;
            border-collapse: collapse;
        }

        th {
            background: var(--bg-secondary);
            padding: 12px;
            text-align: left;
            font-weight: 600;
            border-bottom: 2px solid var(--border-color);
        }

        td {
            padding: 12px;
            border-bottom: 1px solid var(--border-color);
        }

        .badge {
            padding: 4px 10px;
            border-radius: 15px;
            font-size: 11px;
            font-weight: 600;
        }

        .badge-disponivel { background: #00C851; color: white; }
        .badge-solicitado { background: #FFA500; color: white; }
        .badge-pago { background: #17a2b8; color: white; }
        .badge-pendente { background: #FFA500; color: white; }
        .badge-concluido { background: #00C851; color: white; }

        @media (max-width: 768px) {
            .navbar-container {
                flex-direction: column;
                gap: 15px;
            }

            .stats-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>

    <nav class="navbar">
        <div class="navbar-container">
            <div class="navbar-brand">
                <i class="fas fa-wallet"></i>
                Financeiro
            </div>

            <div class="navbar-links">
                <a href="dashboard.php" class="nav-link">
                    <i class="fas fa-home"></i> Dashboard
                </a>
                <a href="produtor_eventos.php" class="nav-link">
                    <i class="fas fa-calendar-alt"></i> Eventos
                </a>
                <a href="produtor_listas.php" class="nav-link">
                    <i class="fas fa-list"></i> Listas
                </a>
                <a href="produtor_financeiro.php" class="nav-link" style="border-color: var(--color-gold);">
                    <i class="fas fa-wallet"></i> Financeiro
                </a>
                <a href="logout.php" class="nav-link" style="border-color: #dc3545; color: #dc3545;">
                    <i class="fas fa-sign-out-alt"></i> Sair
                </a>
            </div>
        </div>
    </nav>

    <div class="container">

        <?php if ($sucesso): ?>
            <div class="alert alert-success"><?= htmlspecialchars($sucesso) ?></div>
        <?php endif; ?>

        <?php if ($erro): ?>
            <div class="alert alert-error"><?= htmlspecialchars($erro) ?></div>
        <?php endif; ?>

        <!-- ESTATÍSTICAS -->
        <div class="stats-grid">
            <div class="stat-card">
                <i class="fas fa-wallet stat-icon"></i>
                <div class="stat-value"><?= formatarMoeda($saldo_disponivel) ?></div>
                <div class="stat-label">Saldo Disponível</div>
            </div>

            <div class="stat-card">
                <i class="fas fa-clock stat-icon"></i>
                <div class="stat-value"><?= formatarMoeda($saldo_solicitado) ?></div>
                <div class="stat-label">Solicitado</div>
            </div>

            <div class="stat-card">
                <i class="fas fa-check-circle stat-icon"></i>
                <div class="stat-value"><?= formatarMoeda($total_recebido) ?></div>
                <div class="stat-label">Total Recebido</div>
            </div>
        </div>

        <!-- SOLICITAR SAQUE -->
        <div class="section">
            <h2 class="section-title">
                <i class="fas fa-money-bill-wave"></i>
                Solicitar Saque PIX
            </h2>

            <form method="POST">
                <div class="form-group">
                    <label class="form-label">Valor</label>
                    <input type="number" name="valor_pix" class="form-input" step="0.01" min="1"
                           max="<?= $saldo_disponivel ?>" placeholder="R$ 0,00" required>
                    <small style="color: var(--text-secondary);">Saldo disponível: <?= formatarMoeda($saldo_disponivel) ?></small>
                </div>

                <div class="form-group">
                    <label class="form-label">Chave PIX</label>
                    <input type="text" name="pix" class="form-input" placeholder="Sua chave PIX" required>
                </div>

                <div class="form-group">
                    <label class="form-label">CPF</label>
                    <input type="text" name="cpf" class="form-input" placeholder="000.000.000-00" required>
                </div>

                <button type="submit" name="solicitar_pix" class="btn btn-primary">
                    <i class="fas fa-paper-plane"></i> Solicitar Saque
                </button>
            </form>
        </div>

        <!-- COMISSÕES POR EVENTO -->
        <?php if (count($comissoes) > 0): ?>
            <div class="section">
                <h2 class="section-title">
                    <i class="fas fa-coins"></i>
                    Comissões por Evento
                </h2>

                <table>
                    <thead>
                        <tr>
                            <th>Evento</th>
                            <th>Data</th>
                            <th>Convidados</th>
                            <th>Check-ins</th>
                            <th>Valor</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($comissoes as $com): ?>
                            <tr>
                                <td><strong><?= htmlspecialchars($com['evento_nome']) ?></strong></td>
                                <td><?= date('d/m/Y', strtotime($com['evento_data'])) ?></td>
                                <td><?= $com['convidados_total'] ?></td>
                                <td><?= $com['convidados_checkin'] ?></td>
                                <td><strong><?= formatarMoeda($com['valor']) ?></strong></td>
                                <td>
                                    <span class="badge badge-<?= $com['status'] ?>">
                                        <?= ucfirst($com['status']) ?>
                                    </span>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>

        <!-- HISTÓRICO DE SAQUES -->
        <?php if (count($saques) > 0): ?>
            <div class="section">
                <h2 class="section-title">
                    <i class="fas fa-history"></i>
                    Histórico de Saques
                </h2>

                <table>
                    <thead>
                        <tr>
                            <th>Data</th>
                            <th>Tipo</th>
                            <th>Valor</th>
                            <th>Chave PIX</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($saques as $saque): ?>
                            <tr>
                                <td><?= date('d/m/Y H:i', strtotime($saque['created_at'])) ?></td>
                                <td><?= $saque['tipo'] === 'pix' ? 'PIX' : 'Crédito' ?></td>
                                <td><strong><?= formatarMoeda($saque['valor']) ?></strong></td>
                                <td><?= htmlspecialchars($saque['pix'] ?? '-') ?></td>
                                <td>
                                    <span class="badge badge-<?= $saque['status'] ?>">
                                        <?= ucfirst($saque['status']) ?>
                                    </span>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>

    </div>

</body>
</html>
